/**************************************************************************************
Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.
***************************************************************************************
$Id: EipObject_Public.h 67315 2017-02-17 16:38:35Z MarcBommert $:

Description:
  Task EthernetIP Object
  Public declaration of task

Changes:
 Date          Description
 -----------------------------------------------------------------------------------
 2010-11-01    created
**************************************************************************************/

/*#####################################################################################*/
/*!
 * \file EipObject_Public.h
 * */

/*#####################################################################################*/
#ifndef __EIPOBJECT_PUBLIC_H
#define __EIPOBJECT_PUBLIC_H

#include "EtherNetIP.h"


/* pragma pack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_PACK_1(EIPOBJECT_PUBLIC)
#endif

/* for internal use */
#define EIP_OBJECT_SET_DEVICEINFO_REQ_DATABASE  0xF0000000

/*#####################################################################################*/
/*#####################################################################################*/
/* common defines */

#define EIP_OBJECT_MAX_IO_DATA_LEN  2000

/*! \ingroup EIP_OBJECT_CIP_SERVICE_REQ_Content
 * \{*/
#define EIP_OBJECT_MAX_PACKET_LEN   1520  /*!< Maximum packet length */
/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/
/* commands */

#define   EIP_OBJECT_ID_GETINFO_REQ                     0x00001A00  /* only for internal use, to get a pointer to the device information */
#define   EIP_OBJECT_ID_GETINFO_CNF                     0x00001A01

/*! \addtogroup EIP_OBJECT_MR_REGISTER_REQ_Content
 * \{*/
#define    EIP_OBJECT_MR_REGISTER_REQ                     0x00001A02  /*!< Register a new object at the message router    */
#define   EIP_OBJECT_MR_REGISTER_CNF                    0x00001A03
/*! \}*/

#define   EIP_OBJECT_OPEN_REQ                           0x00001A08  /* only for internal use, Request to open a Connection */
#define   EIP_OBJECT_OPEN_CNF                           0x00001A09

#define   EIP_OBJECT_CLOSE_REQ                          0x00001A0A /* only for internal use, Request to close a Connection */
#define   EIP_OBJECT_CLOSE_CNF                          0x00001A0B

/*! \addtogroup EIP_OBJECT_AS_REGISTER_REQ_Content
 * \{*/
#define    EIP_OBJECT_AS_REGISTER_REQ                     0x00001A0C  /*!< Register a new Assembly Instance               */
#define   EIP_OBJECT_AS_REGISTER_CNF                    0x00001A0D
/*! \}*/

#define    EIP_OBJECT_CM_OPEN_CONN_REQ                   0x00001A0E  /*!< Open a new connection                          */
#define   EIP_OBJECT_CM_OPEN_CONN_CNF                   0x00001A0F
#define    EIP_OBJECT_CM_CONN_FAULT_IND                   0x00001A12  /*!< Indicate a connection fault                    */
#define   EIP_OBJECT_CM_CONN_FAULT_RES                  0x00001A13
#define    EIP_OBJECT_CM_CLOSE_CONN_REQ                   0x00001A14  /*!< Close a Connection                             */
#define   EIP_OBJECT_CM_CLOSE_CONN_CNF                  0x00001A15

/*! \addtogroup EIP_OBJECT_ID_SETDEVICEINFO_REQ_Content
 * \{*/
#define   EIP_OBJECT_ID_SETDEVICEINFO_REQ               0x00001A16  /*!< Set the device information                     */
#define   EIP_OBJECT_ID_SETDEVICEINFO_CNF               0x00001A17
/*! \}*/

#define   EIP_OBJECT_GET_INPUT_REQ                      0x00001A20  /*!< Getting the latest Input Data                  */
#define   EIP_OBJECT_GET_INPUT_CNF                      0x00001A21

#define   EIP_OBJECT_SET_OUTPUT_REQ                     0x00001A22  /*!< Setting the Output Data                        */
#define   EIP_OBJECT_SET_OUTPUT_CNF                     0x00001A23

/*! \addtogroup EIP_OBJECT_RESET_IND_Content
 * \{*/
#define   EIP_OBJECT_RESET_IND                          0x00001A24  /*!< Indicate a reset request from the device       */
#define   EIP_OBJECT_RESET_RES                          0x00001A25
/*! \}*/
/*! \addtogroup EIP_OBJECT_RESET_REQ_Content
 * \{*/
#define   EIP_OBJECT_RESET_REQ                          0x00001A26  /*!< Request a Reset                                */
#define   EIP_OBJECT_RESET_CNF                          0x00001A27
/*! \}*/

/* Deprecated Command (since V2.6) */
//#define   EIP_OBJECT_TCP_STARTUP_CHANGE_IND             0x00001A28
//#define   EIP_OBJECT_TCP_STARTUP_CHANGE_RES             0x00001A29

/* Deprecated Command (since V2.6) */
//#define   EIP_OBJECT_TCP_STARTUP_CHANGE_REQ             0x00001A2A  /* not Implemented yet */
//#define   EIP_OBJECT_TCP_STARTUP_CHANGE_CNF             0x00001A2B

#define EIP_OBJECT_CHANGE_HOST_TIMEOUT_IND              0x00001A2C
#define EIP_OBJECT_CHANGE_HOST_TIMEOUT_RES              0x00001A2D

/*! \addtogroup EIP_OBJECT_CONNECTION_IND_Content
 * \{*/
#define   EIP_OBJECT_CONNECTION_IND                     0x00001A2E  /*!< Indicate connection state                        */
#define   EIP_OBJECT_CONNECTION_RES                     0x00001A2F
/*! \}*/
/*! \addtogroup EIP_OBJECT_FAULT_IND_Content
 * \{*/
#define   EIP_OBJECT_FAULT_IND                          0x00001A30  /*!< Indicates a fault                           */
#define   EIP_OBJECT_FAULT_RES                          0x00001A31
/*! \}*/
/*! \addtogroup EIP_OBJECT_READY_REQ_Content
 * \{*/
#define   EIP_OBJECT_READY_REQ                          0x00001A32  /*!< Change application ready state                   */
#define   EIP_OBJECT_READY_CNF                          0x00001A33
/*! \}*/

#define   EIP_OBJECT_REGISTER_CONNECTION_REQ            0x00001A34  /*!< Register Connection at Connection Configuration Object */
#define   EIP_OBJECT_REGISTER_CONNECTION_CNF            0x00001A35

#define   EIP_OBJECT_UNCONNECT_MESSAGE_REQ              0x00001A36  /*!< Send an unconnected message request              */
#define   EIP_OBJECT_UNCONNECT_MESSAGE_CNF              0x00001A37

#define   EIP_OBJECT_OPEN_CL3_REQ                       0x00001A38  /*!< Open Class 3 Connection                          */
#define   EIP_OBJECT_OPEN_CL3_CNF                       0x00001A39

#define   EIP_OBJECT_CONNECT_MESSAGE_REQ                0x00001A3A  /*!< Send Class 3 Message Request                     */
#define   EIP_OBJECT_CONNECT_MESSAGE_CNF                0x00001A3B

#define   EIP_OBJECT_CLOSE_CL3_REQ                      0x00001A3C  /*!< Close Class 3 Connection                         */
#define   EIP_OBJECT_CLOSE_CL3_CNF                      0x00001A3D

/*! \addtogroup EIP_OBJECT_CL3_SERVICE_IND_Content
 *  \{ */
#define   EIP_OBJECT_CL3_SERVICE_IND                    0x00001A3E  /*!< Class3 Service Request                           */
#define   EIP_OBJECT_CL3_SERVICE_RES                    0x00001A3F
/*! \}*/
/*! \addtogroup EIP_OBJECT_CONNECTION_CONFIG_IND_Content
 * \{*/
#define   EIP_OBJECT_CONNECTION_CONFIG_IND              0x00001A40  /*!< Indication of configuration data of a connection */
#define   EIP_OBJECT_CONNECTION_CONFIG_RES              0x00001A41
/*! \}*/
/*! \addtogroup EIP_OBJECT_CFG_QOS_REQ_Content
 *  \{ */
#define   EIP_OBJECT_CFG_QOS_REQ                        0x00001A42  /*!< Configure the QoS object                         */
#define   EIP_OBJECT_CFG_QOS_CNF                        0x00001A43
/*! \}*/
/*! \addtogroup EIP_OBJECT_REGISTER_SERVICE_REQ_Content
 *  \{ */
#define   EIP_OBJECT_REGISTER_SERVICE_REQ               0x00001A44  /*!< Register a new service at the message router     */
#define   EIP_OBJECT_REGISTER_SERVICE_CNF               0x00001A45
/*! \}*/
/*! \addtogroup EIP_OBJECT_QOS_CHANGE_IND_Content
 * \{*/
#define   EIP_OBJECT_QOS_CHANGE_IND                     0x00001A46  /*!< Indication of QoS Parameter changes              */
#define   EIP_OBJECT_QOS_CHANGE_RES                     0x00001A47
/*! \}*/

#define   EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ              0x00001A48  /*!< Activate and deactivate slave from scanlist      */
#define   EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF              0x00001A49

#define   EIP_OBJECT_FWD_OPEN_FWD_IND                   0x00001A4A
#define   EIP_OBJECT_FWD_OPEN_FWD_RES                   0x00001A4B

#define   EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND        0x00001A4C
#define   EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_RES        0x00001A4D

#define   EIP_OBJECT_FWD_CLOSE_FWD_IND                  0x00001A4E
#define   EIP_OBJECT_FWD_CLOSE_FWD_RES                  0x00001A4F

#define   EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND           0x00001A50
#define   EIP_OBJECT_CC_FWD_OPEN_RESPONSE_RES           0x00001A51

#define   EIP_OBJECT_AS_STATE_IND                       0x00001A52
#define   EIP_OBJECT_AS_STATE_RES                       0x00001A53

#define   EIP_OBJECT_CM_ABORT_CONNECTION_REQ            0x00001A54
#define   EIP_OBJECT_CM_ABORT_CONNECTION_CNF            0x00001A55

#define   EIP_OBJECT_DELETE_IO_CONFIGURATION_REQ        0x00001A56
#define   EIP_OBJECT_DELETE_IO_CONFIGURATION_CNF        0x00001A57

/*! \addtogroup EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_Content
 *  \{ */
#define   EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ         0x00001A58
#define   EIP_OBJECT_CREATE_OBJECT_TIMESYNC_CNF         0x00001A59
/*! \}*/

#ifdef __USE_LLDP_SNMP

#define   EIP_OBJECT_LLDP_CONTROL_REQ                   0x00001A5A
#define   EIP_OBJECT_LLDP_CONTROL_CNF                   0x00001A5B

#define   EIP_OBJECT_LLDP_CONFIG_REQ                    0x00001A5C
#define   EIP_OBJECT_LLDP_CONFIG_CNF                    0x00001A5D

#endif /* #ifdef __USE_LLDP_SNMP */
/*##############################################################*/

#define   EIP_OBJECT_CYCLIC_EVENT_IND                   0x00001AE0
#define   EIP_OBJECT_WATCHDOG_IND                       0x00001AE2
#define   EIP_OBJECT_CC_KICK_TIME_IND                   0x00001AED
#define   EIP_OBJECT_CC_SCANLIST_TIME_IND               0x00001AEF

/*! \addtogroup EIP_OBJECT_TI_SET_SNN_REQ_Content
 *  \{ */
#define   EIP_OBJECT_TI_SET_SNN_REQ                     0x00001AF0 /*!<  Set the Safety Network Number                    */
#define   EIP_OBJECT_TI_SET_SNN_CNF                     0x00001AF1
/*! \}*/
/*! \addtogroup EIP_OBJECT_SET_PARAMETER_REQ_Content
 *  \{ */
#define   EIP_OBJECT_SET_PARAMETER_REQ                  0x00001AF2
#define   EIP_OBJECT_SET_PARAMETER_CNF                  0x00001AF3
/*! \}*/

#define   EIP_OBJECT_FORWARD_CLOSE_IND                  0x00001AF4
#define   EIP_OBJECT_FORWARD_CLOSE_RES                  0x00001AF5


/*! \addtogroup EIP_OBJECT_CIP_SERVICE_REQ_Content
 * \{ */
#define   EIP_OBJECT_CIP_SERVICE_REQ                    0x00001AF8
#define   EIP_OBJECT_CIP_SERVICE_CNF                    0x00001AF9
/* \} */
/*! \addtogroup EIP_OBJECT_CIP_OBJECT_CHANGE_IND_Content
 * \{ */
#define   EIP_OBJECT_CIP_OBJECT_CHANGE_IND              0x00001AFA
#define   EIP_OBJECT_CIP_OBJECT_CHANGE_RES              0x00001AFB
/* \} */
/*! \addtogroup EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_Content
 * \{ */
#define   EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ  0x00001AFC
#define   EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_CNF  0x00001AFD
/* \} */

/*! \addtogroup EIP_OBJECT_AS_TRIGGER_TYPE_IND_Content
 * \{ */
#define   EIP_OBJECT_AS_TRIGGER_TYPE_IND                0x00001AFE
#define   EIP_OBJECT_AS_TRIGGER_TYPE_RES                0x00001AFF
/* \} */

/*#####################################################################################*/
/*#####################################################################################*/
/* defines of Packet sizes */

#define EIP_OBJECT_CM_OPEN_CONN_REQ_SIZE  (sizeof(EIP_OBJECT_CM_OPEN_CONN_REQ_T))
#define EIP_OBJECT_SET_OUTPUT_REQ_SIZE    (sizeof(EIP_OBJECT_SET_OUTPUT_REQ_T) - EIP_OBJECT_MAX_IO_DATA_LEN)
#define EIP_OBJECT_SET_OUTPUT_CNF_SIZE    (sizeof(EIP_OBJECT_SET_OUTPUT_CNF_T))
#define EIP_OBJECT_GET_INPUT_REQ_SIZE     (sizeof(EIP_OBJECT_GET_INPUT_REQ_T) )
#define EIP_OBJECT_GET_INPUT_CNF_SIZE     (sizeof(EIP_OBJECT_GET_INPUT_CNF_T) - EIP_OBJECT_MAX_IO_DATA_LEN)
#define EIP_OBJECT_FORWARD_CLOSE_IND_SIZE  sizeof(EIP_OBJECT_FORWARD_CLOSE_IND_T) - 1
#define EIP_OBJECT_FORWARD_CLOSE_RES_SIZE  0

/* general structures */

typedef __PACKED_PRE struct EIP_CC_USER_ATTR_DATA_Ttag
{
  TLR_UINT32  ulInstanceOT;
  TLR_UINT32  ulOffsetOT;
  TLR_UINT32  ulInstanceTO;
  TLR_UINT32  ulOffsetTO;
  TLR_UINT32  ulReadyToConnect;
}__PACKED_POST EIP_CC_USER_ATTR_DATA_T;

#define EIP_CC_USER_ATTR_FORMAT   0x1
#define EIP_CC_USER_ATTR_FORMAT_2 0x2
#define EIP_CC_USER_ATTR_LEN        (sizeof(EIP_CC_USER_ATTR_DATA_T) - sizeof(TLR_UINT32))
#define EIP_CC_USER_ATTR_LEN_2      sizeof(EIP_CC_USER_ATTR_DATA_T)

typedef struct EIP_AS_TRIBUF_Ttag
{
  TLR_UINT32 ulReserved;
  TLR_UINT32 ulFlags;
  TLR_UINT32 ulStatus;
  TLR_UINT16 usDataSize;
  TLR_UINT16 usSeqCount;
  TLR_UINT32 ulRunIdle;
  /* TLR_UINT8 abData[]; */
} EIP_AS_TRIBUF_T;

/*#####################################################################################*/
/*#####################################################################################*/
/* Eip Object, packets */

/*! \defgroup EIP_OBJECT_MR_REGISTER_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_MR_REGISTER_REQ
 * \{*/

typedef  __PACKED_PRE struct EIP_OBJECT_MR_REGISTER_REQ_Ttag
{
  TLR_HANDLE hObjectQue;              /*!< Deprecated, set to 0x00 */

  TLR_UINT32 ulClass;                  /*!< Object class identifier

                                           \valueRange \n\n
                                           Instances of the Assembly Object are divided into the following address ranges to \n
                                           provide for extensions to device profiles \n\n
                                           0x0001 - 0x0063 : Open                            \n
                                           0x0064 - 0x00C7 : Vendor Specific                 \n
                                           0x00C8 - 0x00EF : Reserved by ODVA for future use \n
                                           0x00F0 - 0x02FF : Open                            \n
                                           0x0300 - 0x04FF : Vendor Specific                 \n
                                           0x0500 - 0xFFFF : Reserved by ODVA for future use \n */

  TLR_UINT32 ulAccessTyp;              /*!< Reserved, set to 0x00 */

}__PACKED_POST EIP_OBJECT_MR_REGISTER_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_MR_PACKET_REGISTER_REQ_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_MR_REGISTER_REQ_T  tData;
}__PACKED_POST EIP_OBJECT_MR_PACKET_REGISTER_REQ_T;

#define EIP_OBJECT_MR_REGISTER_REQ_SIZE   (sizeof(EIP_OBJECT_MR_REGISTER_REQ_T) )

typedef  __PACKED_PRE struct EIP_OBJECT_MR_PACKET_REGISTER_CNF_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
}__PACKED_POST EIP_OBJECT_MR_PACKET_REGISTER_CNF_T;

#define EIP_OBJECT_MR_REGISTER_CNF_SIZE   0

/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_REGISTER_SERVICE_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_REGISTER_SERVICE_REQ
 * \{*/

typedef  __PACKED_PRE struct EIP_OBJECT_REGISTER_SERVICE_REQ_Ttag
{
  TLR_UINT32 ulService;                /*!< Service code \n

                                      \valueRange \n\n
                                       Instances of the Assembly Object are divided into the following address ranges to \n
                                       provide for extensions to device profiles \n\n

                                       0x00 - 0x31 : Open. These are referred to as Common Services.
                                                     These are defined in the EtherNet/IP spec. Vol.1 Appendix A.\n
                                       0x32 - 0x4A : Vendor Specific                        \n
                                       0x4B - 0x63 : Object Class Specific                  \n
                                       0x64 - 0x7F : Reserved by ODVA for future use        \n
                                       0x80 - 0xFF : Reserved for use as Reply Service Code \n

                                       \description \n\n
                                       Pre-defined service codes (hex): \n\n
                                       00:    Reserved \n
                                       01:    Get_Attributes_All \n
                                       02:    Set_Attributes_All \n
                                       03:    Get_Attribute_List \n
                                       04:    Set_Attribute_List \n
                                       05:    Reset \n
                                       06:    Start \n
                                       07:    Stop \n
                                       08:    Create \n
                                       09:    Delete \n
                                       0A:    Multiple_Service_Packet \n
                                       0B:    Reserved for future use \n
                                       0D:    Apply_Attributes \n
                                       0E:    Get_Attribute_Single \n
                                       0F:    Reserved for future use \n
                                       10:   Set_Attribute_Single \n
                                       11:    Find_Next_Object_Instance \n
                                       12-13: Reserved for future use \n
                                       14:    Error Response (used by DevNet only) \n
                                       15:    Restore \n
                                       16:    Save \n
                                       17:    No Operation (NOP) \n
                                       18:    Get_Member \n
                                       19:    Set_Member \n
                                       1A:    Insert_Member \n
                                       1B:    Remove_Member \n
                                       1C:    GroupSync \n
                                       1D-31: Reserved for additional Common Services \n */


}__PACKED_POST EIP_OBJECT_REGISTER_SERVICE_REQ_T;

/* command for register a new object to the message router */
typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_REGISTER_SERVICE_REQ_Ttag
{
  TLR_PACKET_HEADER_T                tHead;
  EIP_OBJECT_REGISTER_SERVICE_REQ_T  tData;
}__PACKED_POST EIP_OBJECT_PACKET_REGISTER_SERVICE_REQ_T;

#define EIP_OBJECT_REGISTER_SERVICE_REQ_SIZE (sizeof(EIP_OBJECT_REGISTER_SERVICE_REQ_T))

typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_REGISTER_SERVICE_CNF_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
}__PACKED_POST EIP_OBJECT_PACKET_REGISTER_SERVICE_CNF_T;

#define EIP_OBJECT_REGISTER_SERVICE_CNF_SIZE 0

/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ
 * \{*/
#define EIP_OBJECT_CC_SLAVE_ACTIVATE    0x00000001
#define EIP_OBJECT_CC_SLAVE_DEACTIVATE  0x00000002

typedef  __PACKED_PRE struct EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_Ttag
{
  TLR_UINT32 ulSlaveHandle;           /*!< Handle of the connection to the slave \n

                                      \valueRange \n\n

                                      \description \n\n
                                       none \n\n */

  TLR_UINT32 ulActivate;              /*!< Action to the slave \n

                                      \valueRange \n\n
                                       0x01 : activate slave at scanlist             \n
                                       0x02 : deactivate slave from scanlist         \n

                                      \description \n\n
                                       none \n\n */

  TLR_UINT32 ulDelayTime;             /*!< Delay to execute start the communication \n

                                      \valueRange \n\n
                                       0x0000 - 0x7FFF : time at ms \n
                                       0xFFFFFFFF : default time from configuration \n

                                      \description \n\n
                                       none \n\n */

}__PACKED_POST EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_T;

/* command for register a new object to the message router */
typedef  __PACKED_PRE struct EIP_OBJECT_CC_PACKET_SLAVE_ACTIVATE_REQ_Ttag
{
  TLR_PACKET_HEADER_T                 tHead;
  EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_T  tData;
}__PACKED_POST EIP_OBJECT_CC_PACKET_SLAVE_ACTIVATE_REQ_T;

#define EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_SIZE (sizeof(EIP_OBJECT_CC_SLAVE_ACTIVATE_REQ_T))

typedef  __PACKED_PRE struct EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF_Ttag
{
  TLR_UINT32 ulSlaveHandle;           /*!< Handle of the connection to the slave \n

                                      \valueRange \n\n

                                      \description \n\n
                                       none \n\n */

  TLR_UINT32 ulActivate;              /*!< Action to the slave \n

                                      \valueRange \n\n
                                       0x01 : activate slave at scanlist             \n
                                       0x02 : deactivate slave from scanlist         \n

                                      \description \n\n
                                       none \n\n */

}__PACKED_POST EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF_T;

/* command for register a new object to the message router */
typedef  __PACKED_PRE struct EIP_OBJECT_CC_PACKET_SLAVE_ACTIVATE_CNF_Ttag
{
  TLR_PACKET_HEADER_T                 tHead;
  EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF_T  tData;
}__PACKED_POST EIP_OBJECT_CC_PACKET_SLAVE_ACTIVATE_CNF_T;

#define EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF_SIZE (sizeof(EIP_OBJECT_CC_SLAVE_ACTIVATE_CNF_T))

/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/

/* command routes a Data portion to an user installed object */
typedef __PACKED_PRE struct EIP_MSG_ROUTE_USER_Ttag
{
  TLR_PACKET_HEADER_T  tHead;
  TLR_UINT32 ulHandle;                  /* internal handle of the request */
  TLR_UINT16 usLen;                      /* length of the request date      */
  TLR_UINT8 *ptData;                    /* data of the request            */
} __PACKED_POST EIP_MSG_ROUTE_USER_T;

/*#####################################################################################*/
/*#####################################################################################*/

/* EIP_MSG_OPEN_CONNECTION_REQ_T */
typedef  __PACKED_PRE struct EIP_OBJECT_CM_OPEN_CONN_REQ_Ttag
{
  TLR_UINT32    ulConnectionSn;        /* serial number of the connection */
  TLR_UINT32    ulGRC;                 /* Generic Error Code   */
  TLR_UINT32    ulERC;                 /* Extendet Error Code  */

  /* Connection parameters.   */
  TLR_UINT32    ulTimeoutMult;
  TLR_UINT8     bClassTrigger;
  TLR_UINT32    ulProRpi;
  TLR_UINT32    ulProParams;
  TLR_UINT32    ulConRpi;
  TLR_UINT32    ulConParams;

  /* Connection path & module configuration.   */
  TLR_UINT32    ulHostDestSize;   /* length of the hostname/ip */
  TLR_UINT32    ulHostDestOffs;   /* hostname or ip */
  TLR_UINT32    ulProdInhibit;    /* production inhibit time in ms */

  TLR_UINT32    ulPort;

  TLR_UINT32    ulOpenPathSize;
  TLR_UINT32    ulClosePathSize;
  TLR_UINT32    ulPathOffs;

  TLR_UINT32    ulConfig1Size;
  TLR_UINT32    ulConfig1Offs;

  TLR_UINT32    ulConfig2Size;
  TLR_UINT32    ulConfig2Offs;

  /* Data mapping information on our side.   */

  TLR_UINT32    ulClass;
  TLR_UINT32    ulInstance;
  TLR_UINT32    ulProConnPoint;
  TLR_UINT32    ulConConnPoint;

  TLR_UINT32    ulFwdOpenTimeout;
}__PACKED_POST EIP_OBJECT_CM_OPEN_CONN_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_PACKET_OPEN_CONN_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CM_OPEN_CONN_REQ_T tData;
}__PACKED_POST EIP_OBJECT_CM_PACKET_OPEN_CONN_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_OPEN_CONN_CNF_Ttag
{
  TLR_UINT32 ulTransportID;
  TLR_UINT32    ulGRC;                 /* Generic Error Code   */
  TLR_UINT32    ulERC;                 /* Extendet Error Code  */

}__PACKED_POST EIP_OBJECT_CM_OPEN_CONN_CNF_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_PACKET_OPEN_CONN_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CM_OPEN_CONN_CNF_T tData;
}__PACKED_POST EIP_OBJECT_CM_PACKET_OPEN_CONN_CNF_T;

/*#####################################################################################*/
/*#####################################################################################*/

/* EIP_MSG_CLOSE_CONNECTION_REQ_T */
typedef  __PACKED_PRE struct EIP_OBJECT_CM_CLOSE_CONN_REQ_Ttag
{
  TLR_UINT32    ulConnectionSn;
  TLR_UINT32    ulOpenPathSize;
  TLR_UINT32    ulClosePathSize;
  TLR_UINT32    ulPathOffs;
}__PACKED_POST EIP_OBJECT_CM_CLOSE_CONN_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_PACKET_CLOSE_CONN_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CM_CLOSE_CONN_REQ_T tData;
}__PACKED_POST EIP_OBJECT_CM_PACKET_CLOSE_CONN_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_CLOSE_CONN_CNF_Ttag
{
  TLR_UINT32    ulGRC;                 /* Generic Error Code   */
  TLR_UINT32    ulERC;                 /* Extendet Error Code  */

}__PACKED_POST EIP_OBJECT_CM_CLOSE_CONN_CNF_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_PACKET_CLOSE_CONN_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CM_CLOSE_CONN_CNF_T tData;
}__PACKED_POST EIP_OBJECT_CM_PACKET_CLOSE_CONN_CNF_T;

/*#####################################################################################*/
/*#####################################################################################*/

/* EIP OBJECT CM CONN FAULT IND */
typedef  __PACKED_PRE struct EIP_OBJECT_CM_CONN_FAULT_IND_Ttag
{
  TLR_UINT32 ulConnectionSn;
  TLR_UINT32 ulReason;
}__PACKED_POST EIP_OBJECT_CM_CONN_FAULT_IND_T;

typedef  __PACKED_PRE struct EIP_OBJECT_CM_PACKET_CONN_FAULT_IND_Ttag
{
  TLR_PACKET_HEADER_T            tHead;
  EIP_OBJECT_CM_CONN_FAULT_IND_T tData;
}__PACKED_POST EIP_OBJECT_CM_PACKET_CONN_FAULT_IND_T;

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_AS_REGISTER_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_AS_REGISTER_REQ
 * \{ */

/* Flags for the Assembly Object */
#define EIP_AS_FLAG_READONLY            0x00000001 /*!< This flag decides whether the newly registered assembly instance is an
                                                        input or an output assembly (also called input/ouput connection point).
                                                        Output connection points will be mapped automatically into the DPM output data area.
                                                        Input connection points will be mapped automatically into the DPM input data area.\n\n
                                                        If set, the assembly instance serves as input connection point (data received from the network).\n
                                                        If not set, the assembly instance serves as output connection point (data transmitted to the network). */

/* #define EIP_AS_FLAG_ACTIVE           0x00000002 */
/* #define EIP_AS_FLAG_TRIBLEBUF        0x00000004 */

#define EIP_AS_FLAG_RUNIDLE             0x00000008 /*!< If set, the assembly data is modeless (i.e. it does not contain
                                                        run/idle information) */

#define EIP_AS_FLAG_NEWDATA             0x00000010 /*!< The new data flag is used internally and must be set to 0. */

#define EIP_AS_FLAG_CONFIG              0x00000020 /*!< This flag is used to make this assembly a configuration
                                                        assembly. For further information have a look at the command
                                                        \ref EIP_OBJECT_CONNECTION_CONFIG_IND */

#define EIP_AS_FLAG_HOLDLASTSTATE       0x00000040 /*!< This flag decides whether the data that is mapped into the
                                                        corresponding DPM memory area is cleared upon closing of
                                                        the connection or whether the last sent/received data is left
                                                        unchanged in the memory.
                                                        If the bit is set the data will be left unchanged. */

#define EIP_AS_FLAG_FIX_SIZE            0x00000080 /*!< This flag decides whether the assembly instance allows
                                                        a connection to be established with a smaller connection size than
                                                        defined in ulSize or whether only the size defined in ulSize is accepted.
                                                        If the bit is set, the connection size in a ForwardOpen must directly
                                                        correspond to ulSize. */

#define EIP_AS_FLAG_FORWARD_RUNIDLE     0x00000100 /*!< For input assemblies that receive the run/idle header, this flag
                                                        decides whether the run/idle header shall remain in the IO data when being
                                                        written into the triple buffer or DPM. This way the host application
                                                        has the possibility to evaluate the run/idle information on its own.
                                                        If the bit is set the run/idle header will be part of the IO data image. */

#define EIP_AS_FLAG_INVISIBLE           0x00000200 /*!< This flag decides whether the assembly instance can be accessed via
                                                        explicit services from the network. \n\n
                                                        Flag is set: the assembly instance is visible.\n
                                                        Flag is not set: the assembly instance is invisible. */

typedef  __PACKED_PRE struct EIP_OBJECT_AS_REGISTER_REQ_Ttag
{
  TLR_UINT32      ulInstance;       /*!< Assembly instance number \n

                                         \valueRange \n\n
                                         Instances of the Assembly Object are divided into the following address ranges to  \n
                                         provide for extensions to device profiles \n\n
                                         0x00000001 - 0x00000063 : Open (static assemblies defined in device profile)       \n
                                         0x00000064 - 0x000000C7 : Vendor Specific static assemblies and dynamic assemblies \n
                                         0x000000C8 - 0x000002FF : Open (static assemblies defined in device profile)       \n
                                         0x00000300 - 0x000004FF : Vendor Specific static assemblies and dynamic assemblies \n
                                         0x00000500 - 0x000FFFFF : Open (static assemblies defined in device profile)       \n
                                         0x00100000 - 0xFFFFFFFF : Reserved by CIP for future use.

                                         \note \n\n
                                         The instance numbers 192 (hex C0) and 193 (hex C1) are usually used as the dummy connection
                                         points for listen only and input only connections. These values <b>must not </b> be used for new
                                         connection points.\n\n
                                         If loadable firmware is used these values are not adaptable. If not, theses values can be changed
                                         within the startup parameters of the object task
                                         (see structure EIP_OBJECT_PARAMETER_T in EipObject_Functionlist.h).
                                         */

  TLR_UINT32      ulDPMOffset;      /*!< Relative offset of the assembly instance data within the DPM input/output area. */
  TLR_UINT32      ulSize;           /*!< Size of the assembly instance data in bytes */
  TLR_UINT32      ulFlags;          /*!< Flags to configure the assembly instance properties \n

                                         \valueRange \n

                                         - Bit 0: <b> \ref EIP_AS_FLAG_READONLY        </b> \n\n
                                         - Bit 1: <b> \ref EIP_AS_FLAG_ACTIVE          </b> \n\n
                                         - Bit 2: <b> \ref EIP_AS_FLAG_TRIBLEBUF       </b> \n\n
                                         - Bit 3: <b> \ref EIP_AS_FLAG_RUNIDLE         </b> \n\n
                                         - Bit 4: <b> \ref EIP_AS_FLAG_NEWDATA         </b> \n\n
                                         - Bit 5: <b> \ref EIP_AS_FLAG_CONFIG          </b> \n\n
                                         - Bit 6: <b> \ref EIP_AS_FLAG_HOLDLASTSTATE   </b> \n\n
                                         - Bit 7: <b> \ref EIP_AS_FLAG_FIX_SIZE        </b> \n\n
                                         - Bit 8: <b> \ref EIP_AS_FLAG_FORWARD_RUNIDLE </b> \n\n
                                         - Bit 9: <b> \ref EIP_AS_FLAG_INVISIBLE       </b> \n\n */

}__PACKED_POST EIP_OBJECT_AS_REGISTER_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_AS_PACKET_REGISTER_REQ_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_AS_REGISTER_REQ_T  tData;
}__PACKED_POST EIP_OBJECT_AS_PACKET_REGISTER_REQ_T;

#define EIP_OBJECT_AS_REGISTER_REQ_SIZE   (sizeof(EIP_OBJECT_AS_REGISTER_REQ_T) )

typedef  __PACKED_PRE struct EIP_OBJECT_AS_REGISTER_CNF_Ttag
{
  TLR_UINT32  ulInstance;               /*!< Assembly instance number from the request packet              */
  TLR_UINT32  ulDPMOffset;              /*!< DPM offset for the instance data area from the request packet */
  TLR_UINT32  ulSize;                   /*!< Size of the data area from the request packet                 */
  TLR_UINT32  ulFlags;                  /*!< Assembly flags from the request packet                        */
  TLR_HANDLE  hDataBuf;                 /*!< Handle of the triple data buffer                              */
}__PACKED_POST EIP_OBJECT_AS_REGISTER_CNF_T;

typedef  __PACKED_PRE struct EIP_OBJECT_AS_PACKET_REGISTER_CNF_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_AS_REGISTER_CNF_T  tData;
}__PACKED_POST EIP_OBJECT_AS_PACKET_REGISTER_CNF_T;

#define EIP_OBJECT_AS_REGISTER_CNF_SIZE   (sizeof(EIP_OBJECT_AS_REGISTER_CNF_T) )

/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/

typedef  __PACKED_PRE struct OBJECT_PACKET_GENERIC_Ttag
{
  TLR_PACKET_HEADER_T  tHead;
  TLR_UINT8   abData[288];         /* Data */
}__PACKED_POST OBJECT_PACKET_GENERIC_T;

/*#####################################################################################*/
/*#####################################################################################*/

typedef  __PACKED_PRE struct EIP_OBJECT_SET_OUTPUT_REQ_Ttag
{
  TLR_UINT32  ulInstance;
  TLR_UINT8   abOutputData[EIP_OBJECT_MAX_IO_DATA_LEN];
}__PACKED_POST EIP_OBJECT_SET_OUTPUT_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_SET_OUTPUT_REQ_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_SET_OUTPUT_REQ_T   tData;
}__PACKED_POST EIP_OBJECT_PACKET_SET_OUTPUT_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_SET_OUTPUT_CNF_Ttag
{
  TLR_UINT32  ulInstance;
}__PACKED_POST EIP_OBJECT_SET_OUTPUT_CNF_T;

typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_SET_OUTPUT_CNF_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_SET_OUTPUT_CNF_T   tData;
}__PACKED_POST EIP_OBJECT_PACKET_SET_OUTPUT_CNF_T;

/*#####################################################################################*/
/*#####################################################################################*/
typedef  __PACKED_PRE struct EIP_OBJECT_GET_INPUT_REQ_Ttag
{
  TLR_UINT32  ulInstance;
}__PACKED_POST EIP_OBJECT_GET_INPUT_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_GET_INPUT_REQ_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
  EIP_OBJECT_GET_INPUT_REQ_T   tData;
}__PACKED_POST EIP_OBJECT_PACKET_GET_INPUT_REQ_T;

typedef  __PACKED_PRE struct EIP_OBJECT_GET_INPUT_CNF_Ttag
{
  TLR_UINT32  ulInstance;
  TLR_BOOLEAN32 fClearFlag;
  TLR_BOOLEAN32 fNewFlag;
  TLR_UINT8   abInputData[EIP_OBJECT_MAX_IO_DATA_LEN];
}__PACKED_POST EIP_OBJECT_GET_INPUT_CNF_T;

typedef  __PACKED_PRE struct EIP_OBJECT_PACKET_GET_INPUT_CNF_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
  EIP_OBJECT_GET_INPUT_CNF_T   tData;
}__PACKED_POST EIP_OBJECT_PACKET_GET_INPUT_CNF_T;

/*#####################################################################################*/
/*#####################################################################################*/

/*! \defgroup EIP_OBJECT_RESET_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_RESET_IND
 * \{ */
typedef struct EIP_OBJECT_RESET_IND_Ttag
{
  TLR_UINT32 ulDataIdx;                 /*!< Index of the service */
  TLR_UINT32 ulResetTyp;                /*!< Type of the reset \n

                                             \valueRange
                                             0, 1

                                             \description
                                             0: Reset is done emulating power cycling of
                                                the device(default)\n
                                             1: Return as closely as possible to the factory
                                                default configuration. Reset is then done
                                                emulating power cycling of the device.<br>
                                                <b>Note</b>:\n Reset type 1 is not supported
                                                by default. It needs to be enabled separately
                                                using the command EIP_OBJECT_SET_PARAMETER_REQ.*/

}EIP_OBJECT_RESET_IND_T;

typedef struct EIP_OBJECT_PACKET_RESET_IND_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_RESET_IND_T  tData;
}EIP_OBJECT_PACKET_RESET_IND_T;

#define EIP_OBJECT_RESET_IND_SIZE          sizeof(EIP_OBJECT_RESET_IND_T)

typedef struct EIP_OBJECT_PACKET_RESET_RES_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_RESET_RES_T;

#define EIP_OBJECT_RESET_RES_SIZE          0

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_RESET_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_RESET_REQ
 * \{ */

/* Reset Modes */
typedef enum
{
  EIP_SYSTEM_RESET = 0,           /*!< 0: Reset is done emulating power cycling of the device (default).
                                          Configuration is not touched                                  */
  EIP_FACTORYDEFAULTS_RESET,      /*!< 1: Reset is done emulating power cycling of the device and
                                          additionally sets configuration back to factory defaults      */
  EIP_CLEAR_WATCHDOG_RESET        /*!< 2: Clears a watch dog error.
                                          In case a watchdog error occurred the stack stops at a specific
                                          point and does not go into normal operation anymore.
                                          Using this type of reset clears this state and the stack starts over
                                          again. */
} EIP_OBJECT_RESET_MODE_E ;

typedef struct EIP_OBJECT_RESET_REQ_Ttag
{
  TLR_UINT32 ulDataIdx;               /*!< Reserved (set to 0) */
  TLR_UINT32 ulResetMode;             /*!< Mode of the reset (see \ref EIP_OBJECT_RESET_MODE_E)   */
}EIP_OBJECT_RESET_REQ_T;

typedef struct EIP_OBJECT_PACKET_RESET_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_RESET_REQ_T  tData;
}EIP_OBJECT_PACKET_RESET_REQ_T;

#define EIP_OBJECT_RESET_REQ_SIZE          sizeof(EIP_OBJECT_RESET_REQ_T)

typedef struct EIP_OBJECT_PACKET_RESET_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_RESET_CNF_T;

#define EIP_OBJECT_RESET_CNF_SIZE          0

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_FAULT_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_FAULT_IND
 * \{ */

typedef struct EIP_OBJECT_PACKET_FAULT_IND_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_FAULT_IND_T;

#define EIP_OBJECT_FAULT_IND_SIZE 0

typedef struct EIP_OBJECT_PACKET_FAULT_RES_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_FAULT_RES_T;

#define EIP_OBJECT_FAULT_RES_SIZE 0
/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CONNECTION_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_CONNECTION_IND
 * \{ */

/* Connection Indication defines *********************************************/
//#define EIP_UNCONNECT  0
//#define EIP_CONNECTED  1
//#define EIP_CONNECTED  2

typedef enum EIP_CONNECTION_STATE_Ttag
{
  EIP_UNCONNECT = 0,   /*!< 0: Connection was closed                  */
  EIP_CONNECTED,       /*!< 1: Connection was established             */
  EIP_DIAGNOSTIC       /*!< 2: Diagnostics (EtherNet/IP Scanner only) */
}EIP_CONNECTION_STATE_T;

typedef enum EIP_EXT_CONNECTION_STATE_Ttag
{
  EIP_CONN_STATE_UNDEFINED = 0, /*!< 0: Undefined, not used  */
  EIP_CONN_STATE_TIMEOUT        /*!< 1: Connection timed out */
}EIP_EXT_CONNECTION_STATE_T;


/* Connection type */
typedef enum EIP_CONNECTION_TYPE_Ttag{
  EIP_CONN_TYPE_UNDEFINED = 0,              /*!< 0: Undefined, not used                                 */
  EIP_CONN_TYPE_CLASS_0_1_EXCLUSIVE_OWNER,  /*!< 1: Class0/Class1 (implicit) exclusive owner connection */
  EIP_CONN_TYPE_CLASS_0_1_REDUNDANT_OWNER,  /*!< 2: Class0/Class1 (implicit) redundant owner connection */
  EIP_CONN_TYPE_CLASS_0_1_LISTEN_ONLY,      /*!< 3: Class0/Class1 (implicit) listen only connection     */
  EIP_CONN_TYPE_CLASS_0_1_INPUT_ONLY,       /*!< 4: Class0/Class1 (implicit) input only connection      */
  EIP_CONN_TYPE_CLASS_3                     /*!< 5: Class3        (explicit) connection                 */
}EIP_CONNECTION_TYPE_T;


typedef struct EIP_OBJECT_OT_CONNECTION_Ttag
{
    TLR_UINT32 ulConnHandle;                /*!< Connection handle
                                               \description
                                               This connection handle is the handle
                                               that was provided when the connection was
                                               registered using \ref EIP_OBJECT_REGISTER_CONNECTION_REQ. */

    TLR_UINT32 ulReserved[4];               /*!< Reserved */
} EIP_OBJECT_OT_CONNECTION_T;

typedef struct EIP_OBJECT_TO_CONNECTION_Ttag
{
    TLR_UINT32 ulClass;                     /*!< CIP object the connection was addressed to              */
    TLR_UINT32 ulInstance;                  /*!< Instance of CIP object                                  */
    TLR_UINT32 ulOTConnPoint;               /*!< O2T connection point                                    */
    TLR_UINT32 ulTOConnPoint;               /*!< T2O connection point                                    */
    TLR_UINT32 ulConnectionType;            /*!< Type of connection ( see \ref EIP_CONNECTION_TYPE_Ttag) */
} EIP_OBJECT_TO_CONNECTION_T;

typedef union EIP_OBJECT_CONNECTION_Ttag /*!< If  is true use tOTConnection otherwise
                                              tTOConnection */
{
    EIP_OBJECT_OT_CONNECTION_T tOTConnection; /*!< Use this structure if ulOriginator is true */
    EIP_OBJECT_TO_CONNECTION_T tTOConnection; /*!< Use this structure if ulOriginator is false */
} EIP_OBJECT_CONNECTION_T;

typedef __PACKED_PRE struct EIP_OBJECT_EXT_CONNECTION_INFO_Ttag
{
  TLR_UINT32  ulProConnId;           /*!< Producer Connection ID (T->O)                 */
  TLR_UINT32  ulConConnId;           /*!< Consumer Connection ID (O->T)                 */

  TLR_UINT32  ulConnSerialNum;       /*!< Connection serial number                      */
  TLR_UINT16  usOrigVendorId;        /*!< Originator device vendor ID                   */
  TLR_UINT32  ulOrigDeviceSn;        /*!< Originator device serial number               */

  /* Producer parameters */
  TLR_UINT32  ulProApi;              /*!< Actual packet interval (usecs) (T->O)         */
  TLR_UINT16  usProConnParams;       /*!< Connection parameters (T->O) from ForwardOpen */

  /* Consumer parameters */
  TLR_UINT32  ulConApi;              /*!< Actual packet interval (usecs) (O->T)          */
  TLR_UINT16  usConConnParams;       /*!< Connection parameters (O->T)  from ForwardOpen */

  TLR_UINT8   bTimeoutMultiplier;    /*!< Connection timeout multiplier                  */

} __PACKED_POST EIP_OBJECT_EXT_CONNECTION_INFO_T;

typedef __PACKED_PRE struct EIP_OBJECT_CONNECTION_IND_Ttag
{
  TLR_UINT32 ulConnectionState;        /*!< Reason of changing the connection state
                                            (see \ref EIP_CONNECTION_STATE_Ttag)                              */
  TLR_UINT32 ulConnectionCount;        /*!< Number of established connections                                 */
  TLR_UINT32 ulOutConnectionCount;     /*!< EtherNet/IP Adapter: Not supported \n
                                            EtherNet/IP Scanner: Number of active originate connections       */
  TLR_UINT32 ulConfiguredCount;        /*!< EtherNet/IP Adapter: Not supported \n
                                            EtherNet/IP Scanner: Number of configured connections             */
  TLR_UINT32 ulActiveCount;            /*!< EtherNet/IP Adapter: Not supported \n
                                            EtherNet/IP Scanner: Number of active connections                 */
  TLR_UINT32 ulDiagnosticCount;        /*!< EtherNet/IP Adapter: Not supported \n
                                            EtherNet/IP Scanner: Number of diagnostic connections             */

  TLR_UINT32 ulOrginator;              /*!< EtherNet/IP Adapter: always 0 \n
                                            EtherNet/IP Scanner: 0 for incoming connection,
                                                                 1 for outgoing connection                    */
  EIP_OBJECT_CONNECTION_T tConnection; /*!< Gives extended information concerning
                                            the connection state (ulConnectionState)                          */
  TLR_UINT32 ulExtendedState;          /*!< Extended status (see \ref EIP_EXT_CONNECTION_STATE_Ttag)          */

  EIP_OBJECT_EXT_CONNECTION_INFO_T tExtInfo; /*!< Additional connection information for
                                                  incoming connections (ulOriginator == 0)                    */

} __PACKED_POST EIP_OBJECT_CONNECTION_IND_T;

typedef struct EIP_OBJECT_PACKET_CONNECTION_IND_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
  EIP_OBJECT_CONNECTION_IND_T  tData;
}EIP_OBJECT_PACKET_CONNECTION_IND_T;

typedef struct EIP_OBJECT_PACKET_CONNECTION_RES_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_CONNECTION_RES_T;

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_READY_REQ_Content Command related contents
 *  \ingroup EIP_OBJECT_READY_REQ
 * \{ */

typedef struct EIP_OBJECT_READY_REQ_Ttag
{
  TLR_UINT32 ulReady;           /*!< Ready state of the application
                                     (starts/stops cyclic communication)\n\n
                                     \valueRange
                                     0, 1
                                     \description
                                     0: Sets application state to "not ready".
                                        Cyclic communication is disabled. All
                                        running I/O connections will be terminated.\n\n
                                     1: Sets application state to "ready".
                                        Cyclic communication is enabled.           */

  TLR_UINT32 ulRunIdle;         /*!< Run/Idle state of the application
                                     (sets the run/idle bit in the run/idle header
                                      for cyclic I/O connections, if used )\n\n
                                     \valueRange
                                     0, 1
                                     \description
                                     0: Sets run/idle state to "idle".\n\n
                                     1: Sets run/idle state to "run".              */
}EIP_OBJECT_READY_REQ_T;

typedef struct EIP_OBJECT_PACKET_READY_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_READY_REQ_T  tData;
}EIP_OBJECT_PACKET_READY_REQ_T;

#define EIP_OBJECT_READY_REQ_SIZE          sizeof(EIP_OBJECT_READY_REQ_T)

typedef struct EIP_OBJECT_PACKET_READY_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_READY_CNF_T;

#define EIP_OBJECT_READY_CNF_SIZE          0

/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/

#define EIP_PATH_LEN             256
#define EIP_CONNECTION_NAME_LEN  64
#define EIP_CONFIG1_LEN          64
#define EIP_CONFIG2_LEN          64
#define EIP_IMPL_DEF_LEN         64


typedef __PACKED_PRE struct EIP_OBJECT_REGISTER_CONNECTION_REQ_Ttag
{
  TLR_UINT8   bGeneralStatus;
  TLR_UINT8   bReserved1;
  TLR_UINT16  usExtendedStatus;
  TLR_UINT16  usConnectionFlags;
  TLR_UINT16  usVendorID;
  TLR_UINT16  usProductType;
  TLR_UINT16  usProductCode;
  TLR_UINT8   bMinRev;
  TLR_UINT8   bMajRev;
  TLR_UINT32  ulCSDataIdx;
  TLR_UINT8   bConnMultiplier;
  TLR_UINT8   bClassTrigger;
  TLR_UINT32  ulRpiOT;
  TLR_UINT16  usNetParamOT;
  TLR_UINT32  ulRpiTO;
  TLR_UINT16  usNetParamTO;
  TLR_UINT8   bOpenPathSize;
  TLR_UINT8   bReserved2;
  TLR_UINT32  ulPathOffset;     /* TLR_UINT8   abConnPath[EIP_PATH_LEN];   */
  TLR_UINT16  usConfig1Size;
  TLR_UINT32  ulConfig1Offset;  /* TLR_UINT8   abConfig1[EIP_CONFIG1_LEN]; */
  TLR_UINT8   bNameSize;
  TLR_UINT8   bReserved3;
  TLR_UINT16  usConnNameLen;
  TLR_UINT8   abConnName[EIP_CONNECTION_NAME_LEN];
  TLR_UINT16  usFormatNumber;
  TLR_UINT8   bImpDefSize;
  TLR_UINT16  usImpDefLen;
  TLR_UINT8   abImpDef[EIP_IMPL_DEF_LEN];
  TLR_UINT16  usConfig2Size;
  TLR_UINT32  ulConfig2Offset;  /* TLR_UINT8   abConfig2[EIP_CONFIG2_LEN]; */
  TLR_UINT16  usProxyVendorID;
  TLR_UINT16  usProxyProductType;
  TLR_UINT16  usProxyProductCode;
  TLR_UINT8   bProxyMinRev;
  TLR_UINT8   bProxyMajRev;
  TLR_UINT32  ulConnHandle;    /* unique handle generated by the configuration tool */
}__PACKED_POST EIP_OBJECT_REGISTER_CONNECTION_REQ_T;

typedef struct EIP_OBJECT_PACKET_REGISTER_CONNECTION_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_REGISTER_CONNECTION_REQ_T  tData;
}EIP_OBJECT_PACKET_REGISTER_CONNECTION_REQ_T;

typedef struct EIP_OBJECT_PACKET_REGISTER_CONNECTION_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
}EIP_OBJECT_PACKET_REGISTER_CONNECTION_CNF_T;

#define EIP_OBJECT_REGISTER_CONNECTION_REQ_SIZE   sizeof(EIP_OBJECT_REGISTER_CONNECTION_REQ_T)
#define EIP_OBJECT_REGISTER_CONNECTION_CNF_SIZE   0;

/*#####################################################################################*/
/*#####################################################################################*/

typedef __PACKED_PRE struct EIP_OBJECT_UNCONNECT_MESSAGE_REQ_Ttag
{
  TLR_UINT32 ulIPAddr;                  /*!< Destination IP address */
  TLR_UINT8  bService;                  /*!< CIP service code       */
  TLR_UINT8  bReserved;                 /*!< Reserved padding       */
  TLR_UINT16 usClass;                   /*!< CIP class ID           */
  TLR_UINT16 usInstance;                /*!< CIP Instance           */
  TLR_UINT16 usAttribute;               /*!< CIP Attribute          */

  TLR_UINT8  abData[EIP_OBJECT_MAX_PACKET_LEN]; /*!< Service Data   */
}__PACKED_POST EIP_OBJECT_UNCONNECT_MESSAGE_REQ_T;


typedef struct EIP_OBJECT_PACKET_UNCONNECT_MESSAGE_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_UNCONNECT_MESSAGE_REQ_T  tData;
}EIP_OBJECT_PACKET_UNCONNECT_MESSAGE_REQ_T;


typedef __PACKED_PRE struct EIP_OBJECT_UNCONNECT_MESSAGE_CNF_Ttag
{
  TLR_UINT32 ulIPAddr;                  /*!< Destination IP address */
  TLR_UINT8  bService;                  /*!< CIP service code       */
  TLR_UINT8  bReserved;                 /*!< Reserved padding       */
  TLR_UINT16 usClass;                   /*!< CIP class ID           */
  TLR_UINT16 usInstance;                /*!< CIP Instance           */
  TLR_UINT16 usAttribute;               /*!< CIP Attribute          */

  TLR_UINT8  abData[EIP_OBJECT_MAX_PACKET_LEN]; /*!< Service Data   */
}__PACKED_POST EIP_OBJECT_UNCONNECT_MESSAGE_CNF_T;


typedef struct EIP_OBJECT_PACKET_UNCONNECT_MESSAGE_CNF_Ttag
{
  TLR_PACKET_HEADER_T                 tHead;
  EIP_OBJECT_UNCONNECT_MESSAGE_CNF_T  tData;
}EIP_OBJECT_PACKET_UNCONNECT_MESSAGE_CNF_T;


#define EIP_OBJECT_UNCONNECT_MESSAGE_REQ_SIZE (sizeof(EIP_OBJECT_UNCONNECT_MESSAGE_REQ_T) - EIP_OBJECT_MAX_PACKET_LEN)
#define EIP_OBJECT_UNCONNECT_MESSAGE_CNF_SIZE (sizeof(EIP_OBJECT_UNCONNECT_MESSAGE_CNF_T) - EIP_OBJECT_MAX_PACKET_LEN)

/*#####################################################################################*/
/*#####################################################################################*/

typedef __PACKED_PRE struct EIP_OBJECT_OPEN_CL3_REQ_Ttag
{
  TLR_UINT32 ulIPAddr;                  /*!< Destination IP address */
  TLR_UINT32 ulTime;                    /*!< Expected Message Time  */
  TLR_UINT32 ulTimeoutMult;             /*!< Timeout Multiplier     */
}__PACKED_POST EIP_OBJECT_OPEN_CL3_REQ_T;


typedef struct EIP_OBJECT_PACKET_OPEN_CL3_REQ_Ttag
{
  TLR_PACKET_HEADER_T         tHead;
  EIP_OBJECT_OPEN_CL3_REQ_T   tData;
}EIP_OBJECT_PACKET_OPEN_CL3_REQ_T;


typedef __PACKED_PRE struct EIP_OBJECT_OPEN_CL3_CNF_Ttag
{
  TLR_UINT32    ulConnection;          /*!< Connection Handle    */
  TLR_UINT32    ulGRC;                 /*!< Generic Error Code   */
  TLR_UINT32    ulERC;                 /*!< Extendet Error Code  */
}__PACKED_POST EIP_OBJECT_OPEN_CL3_CNF_T;


typedef struct EIP_OBJECT_PACKET_OPEN_CL3_CNF_Ttag
{
  TLR_PACKET_HEADER_T         tHead;
  EIP_OBJECT_OPEN_CL3_CNF_T   tData;
}EIP_OBJECT_PACKET_OPEN_CL3_CNF_T;

#define EIP_OBJECT_OPEN_CL3_REQ_SIZE (sizeof(EIP_OBJECT_OPEN_CL3_REQ_T))
#define EIP_OBJECT_OPEN_CL3_CNF_SIZE (sizeof(EIP_OBJECT_OPEN_CL3_CNF_T))

/*#####################################################################################*/
/*#####################################################################################*/

typedef __PACKED_PRE struct EIP_OBJECT_CONNECT_MESSAGE_REQ_Ttag
{
  TLR_UINT32 ulConnection;              /*!< Connection Handle      */
  TLR_UINT8  bService;                  /*!< CIP service code       */
  TLR_UINT8  bReserved;                 /*!< Reserved padding       */
  TLR_UINT16 usClass;                   /*!< CIP class ID           */
  TLR_UINT16 usInstance;                /*!< CIP Instance           */
  TLR_UINT16 usAttribute;               /*!< CIP Attribute          */

  TLR_UINT8  abData[EIP_OBJECT_MAX_PACKET_LEN]; /*!< Service Data   */
}__PACKED_POST EIP_OBJECT_CONNECT_MESSAGE_REQ_T;


typedef struct EIP_OBJECT_PACKET_CONNECT_MESSAGE_REQ_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CONNECT_MESSAGE_REQ_T  tData;
}EIP_OBJECT_PACKET_CONNECT_MESSAGE_REQ_T;


typedef __PACKED_PRE struct EIP_OBJECT_CONNECT_MESSAGE_CNF_Ttag
{
  TLR_UINT32 ulConnection;              /*!< Connection Handle      */
  TLR_UINT8  bService;                  /*!< CIP service code       */
  TLR_UINT8  bReserved;                 /*!< Reserved padding       */
  TLR_UINT16 usClass;                   /*!< CIP class ID           */
  TLR_UINT16 usInstance;                /*!< CIP Instance           */
  TLR_UINT16 usAttribute;               /*!< CIP Attribute          */

  TLR_UINT8  abData[EIP_OBJECT_MAX_PACKET_LEN]; /*!< Service Data   */
}__PACKED_POST EIP_OBJECT_CONNECT_MESSAGE_CNF_T;


typedef struct EIP_OBJECT_PACKET_CONNECT_MESSAGE_CNF_Ttag
{
  TLR_PACKET_HEADER_T     tHead;
  EIP_OBJECT_CONNECT_MESSAGE_CNF_T  tData;
}EIP_OBJECT_PACKET_CONNECT_MESSAGE_CNF_T;


#define EIP_OBJECT_CONNECT_MESSAGE_REQ_SIZE (sizeof(EIP_OBJECT_CONNECT_MESSAGE_REQ_T) - EIP_OBJECT_MAX_PACKET_LEN)
#define EIP_OBJECT_CONNECT_MESSAGE_CNF_SIZE (sizeof(EIP_OBJECT_CONNECT_MESSAGE_CNF_T) - EIP_OBJECT_MAX_PACKET_LEN)

/*#####################################################################################*/
/*#####################################################################################*/

typedef __PACKED_PRE struct EIP_OBJECT_CLOSE_CL3_REQ_Ttag
{
  TLR_UINT32    ulConnection;          /*!< Connection Handle    */
}__PACKED_POST EIP_OBJECT_CLOSE_CL3_REQ_T;


typedef struct EIP_OBJECT_PACKET_CLOSE_CL3_REQ_Ttag
{
  TLR_PACKET_HEADER_T         tHead;
  EIP_OBJECT_CLOSE_CL3_REQ_T   tData;
}EIP_OBJECT_PACKET_CLOSE_CL3_REQ_T;


typedef __PACKED_PRE struct EIP_OBJECT_CLOSE_CL3_CNF_Ttag
{
  TLR_UINT32    ulGRC;                 /*!< Generic Error Code   */
  TLR_UINT32    ulERC;                 /*!< Extendet Error Code  */
}__PACKED_POST EIP_OBJECT_CLOSE_CL3_CNF_T;


typedef struct EIP_OBJECT_PACKET_CLOSE_CL3_CNF_Ttag
{
  TLR_PACKET_HEADER_T         tHead;
  EIP_OBJECT_CLOSE_CL3_CNF_T   tData;
}EIP_OBJECT_PACKET_CLOSE_CL3_CNF_T;

#define EIP_OBJECT_CLOSE_CL3_REQ_SIZE (sizeof(EIP_OBJECT_CLOSE_CL3_REQ_T))
#define EIP_OBJECT_CLOSE_CL3_CNF_SIZE (sizeof(EIP_OBJECT_CLOSE_CL3_CNF_T))

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CL3_SERVICE_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_CL3_SERVICE_IND
 * \{ */

typedef __PACKED_PRE struct EIP_OBJECT_CL3_SERVICE_IND_Ttag
{
  TLR_UINT32    ulConnectionId;          /*!< Connection Handle \n

                                           \description \n
                                           The connection handle uniquely identifies
                                           the connection. The same connection handle is
                                           used for all transmissions of a particular
                                           connection. \n\n
                                           <b>Important: \n</b>
                                           This handle must be used for the response to
                                           this indication */

  TLR_UINT32    ulService;               /*!< Service \n

                                           \description \n
                                           This parameter holds the requested CIP service
                                           that related to the next parameter (ulObject).\n\n
                                           <b>Important: \n</b>
                                           This parameter must be used again for the response
                                           to this indication */

  TLR_UINT32    ulObject;                /*!< Object class \n

                                           \description \n
                                           This parameter holds the CIP object class ID.\n\n
                                           <b>Important: \n</b>
                                           This parameter must be used again for the response
                                           to this indication */

  TLR_UINT32    ulInstance;              /*!< Instance \n

                                           \description \n
                                           This parameter holds the instance number
                                           of the object class specified in ulObject.\n\n
                                           <b>Important: \n</b>
                                           This parameter must be used again for the response
                                           to this indication */

  TLR_UINT32    ulAttribute;             /*!< Attribute \n

                                           \description \n
                                           This parameter holds the attribute number
                                           of the object class instance specified in
                                           ulInstance.\n\n
                                           <b>Important: \n</b>
                                           This parameter must be used again for the response
                                           to this indication */

  TLR_UINT8     abData[1];
}__PACKED_POST EIP_OBJECT_CL3_SERVICE_IND_T;


typedef struct EIP_OBJECT_PACKET_CL3_SERVICE_IND_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
  EIP_OBJECT_CL3_SERVICE_IND_T    tData;
}EIP_OBJECT_PACKET_CL3_SERVICE_IND_T;

#define EIP_OBJECT_CL3_SERVICE_IND_SIZE (sizeof(EIP_OBJECT_CL3_SERVICE_IND_T)-1)


typedef __PACKED_PRE struct EIP_OBJECT_CL3_SERVICE_RES_Ttag
{
  TLR_UINT32    ulConnectionId;          /*!< Connection Handle \n

                                           \description \n
                                           The connection handle from the indication packet. */

  TLR_UINT32    ulService;               /*!< Service \n

                                           \description \n
                                           The service code from the indication packet. */

  TLR_UINT32    ulObject;                /*!< Object class \n

                                           \description \n
                                           The object class ID from the indication packet */

  TLR_UINT32    ulInstance;              /*!< Instance \n

                                           \description \n
                                           The instance number from the indication packet */

  TLR_UINT32    ulAttribute;             /*!< Attribute \n

                                           \description \n
                                           The attribute number from the indication packet */

  TLR_UINT32    ulGRC;                 /*!< Generic Error Code \n

                                           \description \n
                                            This generic error code can be used to
                                            indicate whether the service request was successful
                                            or unsuccessful. */

  TLR_UINT32    ulERC;                 /*!< Extended Error Code \n

                                           \description \n
                                           This extended error code can be used to
                                           describe the occurred error (GRC) in more detail. */

  TLR_UINT8     abData[1];             /*!< Response service  data */

}__PACKED_POST EIP_OBJECT_CL3_SERVICE_RES_T;


typedef struct EIP_OBJECT_PACKET_CL3_SERVICE_RES_Ttag
{
  TLR_PACKET_HEADER_T            tHead;
  EIP_OBJECT_CL3_SERVICE_RES_T   tData;
}EIP_OBJECT_PACKET_CL3_SERVICE_RES_T;


#define EIP_OBJECT_CL3_SERVICE_RES_SIZE (sizeof(EIP_OBJECT_CL3_SERVICE_RES_T)-1)
/*! \}*/
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CONNECTION_CONFIG_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_CONNECTION_CONFIG_IND
 * \{ */

/* EIP_OBJECT_CONNECTION_CONFIG_IND */
typedef __PACKED_PRE struct EIP_OBJECT_CONNECTION_CONFIG_IND_Ttag
{
  TLR_UINT32    ulConnectionId;          /*!< Connection Handle:\n
                                              \description \n
                                              This variable contains the connection handle
                                              that is used by the protocol stack and must
                                              not be changed.                                    */

  TLR_UINT32    ulOTParameter;           /*!< Originator to target parameters:\n
                                              \description \n
                                              This variable contains the connection
                                              parameter for the originator-to-target
                                              direction of the connection. It follows
                                              the rules for network connection parameters
                                              as specified in section 3-5.5.1.1
                                              "Network Connection Parameters" of the
                                              "The CIP Networks Library, Volume 1" document.     */

  TLR_UINT32    ulOTRpi;                 /*!< Originator to target RPI:\n
                                              \description \n
                                              This variable contains the requested packet
                                              interval (RPI) for the originator-to-target
                                              direction of the connection. The time is
                                              specified in microseconds.                         */

  TLR_UINT32    ulOTConnPoint;           /*!< Originator to target (produced) connection point:\n
                                              \description \n
                                              This variable contains the connection point for
                                              originator-to-target direction. It should match
                                              one of the input assembly instances
                                              (flag EIP_AS_FLAG_READONLY set) that were
                                              registered (\ref EIP_OBJECT_AS_REGISTER_REQ)
                                              during the configuration process.                  */

  TLR_UINT32    ulTOParameter;           /*!< Target to originator parameters:\n
                                              \description \n
                                              This variable contains the connection
                                              parameter for the target-to-originator
                                              direction of the connection. It follows
                                              the rules for network connection parameters
                                              as specified in section 3-5.5.1.1
                                              "Network Connection Parameters" of the
                                              "The CIP Networks Library, Volume 1" document.     */

  TLR_UINT32    ulTORpi;                 /*!< Target to originator RPI:\n
                                              \description \n
                                              This variable contains the requested packet
                                              interval (RPI) for the target-to-originator
                                              direction of the connection. The time is
                                              specified in microseconds.                         */

  TLR_UINT32    ulTOConnPoint;           /*!< Target to originator (consumed) connection point:\n
                                              \description \n
                                              This variable contains the connection point for the
                                              target-to-originator direction. It should match one
                                              of the input assembly instances
                                              (flag EIP_AS_FLAG_READONLY not set) that were
                                              registered (\ref EIP_OBJECT_AS_REGISTER_REQ)
                                              during the configuration process.*/

  TLR_UINT32    ulCfgConnPoint;          /*!< Configuration connection point:\n
                                              \description \n
                                              This variable contains the connection point for
                                              the configuration data. It should match one
                                              of the configuration assembly instances
                                              (flag EIP_AS_FLAG_CONFIG set) that were
                                              registered (\ref EIP_OBJECT_AS_REGISTER_REQ)
                                              during the configuration process.                  */

  TLR_UINT8     abData[1];               /*!< Configuration data:\n
                                              \description \n
                                              This byte array includes the configuration data.
                                              The size of the data is included in the field
                                              ulLen in the packet header. Usually, the default
                                              data is the data or safety segment that was received
                                              via a Forward_Open message.\n\n
                                              However, there is a
                                              possibility to receive the entire Forward_open
                                              segment. This can be achieved by setting the
                                              corresponding flag via the
                                              \ref EIP_OBJECT_SET_PARAMETER_REQ command.         */
}__PACKED_POST EIP_OBJECT_CONNECTION_CONFIG_IND_T;


typedef struct EIP_OBJECT_PACKET_CONNECTION_CONFIG_IND_Ttag
{
  TLR_PACKET_HEADER_T                   tHead;
  EIP_OBJECT_CONNECTION_CONFIG_IND_T    tData;
}EIP_OBJECT_PACKET_CONNECTION_CONFIG_IND_T;


typedef __PACKED_PRE struct EIP_OBJECT_CONNECTION_CONFIG_RES_Ttag
{
  TLR_UINT32    ulConnectionId;        /*!< Unchanged connection handle from indication packet                   */
  TLR_UINT32    ulGRC;                 /*!< General Error Code                                                   */
  TLR_UINT32    ulERC;                 /*!< Extended Error Code                                                  */
  TLR_UINT8     abData[1];             /*!< Data that will be sent with the Forward_Open_Response\n\n

                                            \description \n
                                            This could be either "Application Reply data" in case the
                                            packet response is sent back to the stack with tHead.ulSta == 0.\n\n

                                            Or this could be additional status data in case the
                                            packet response is sent back to the stack with tHead.ulSta != 0.\n\n */

}__PACKED_POST EIP_OBJECT_CONNECTION_CONFIG_RES_T;


typedef struct EIP_OBJECT_PACKET_CONNECTION_CONFIG_RES_Ttag
{
  TLR_PACKET_HEADER_T                  tHead;
  EIP_OBJECT_CONNECTION_CONFIG_RES_T   tData;
}EIP_OBJECT_PACKET_CONNECTION_CONFIG_RES_T;

#define EIP_OBJECT_CONNECTION_CONFIG_IND_SIZE (sizeof(EIP_OBJECT_CONNECTION_CONFIG_IND_T)-1) /* - sizeof(abData[1]) */
#define EIP_OBJECT_CONNECTION_CONFIG_RES_SIZE (sizeof(EIP_OBJECT_CONNECTION_CONFIG_RES_T)-1) /* - sizeof(abData[1]) */

/*! \}*/

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CFG_QOS_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_CFG_QOS_REQ
 *  \{ */
#define EIP_OBJECT_QOS_FLAGS_ENABLE          0x00000001 /*!< Enables the QoS object */
#define EIP_OBJECT_QOS_FLAGS_DEFAULT         0x00000002 /*!< Set the default attribute values
                                                             of the QoS object as they are defined
                                                             in the EtherNet/IP specification */
#define EIP_OBJECT_QOS_FLAGS_DISABLE_802_1Q  0x00000004 /*!< Disables attribute 1 of the QoS
                                                             object */

typedef __PACKED_PRE struct EIP_OBJECT_CFG_QOS_REQ_Ttag
{
  TLR_UINT32    ulQoSFlags;                 /*!< Configuration flags \n

                                                 \valueRange
                                                 0...7

                                                 \description
                                                 Enables or disables sending 802.1Q frames on CIP
                                                 messages\n\n
                                                 Bit 0 (EIP_OBJECT_QOS_FLAGS_ENABLE) :\n
                                                 Enables the QoS object\n\n
                                                 Bit 1 (EIP_OBJECT_QOS_FLAGS_DEFAULT):\n
                                                 If set, the stack automatically sets the default values for
                                                 all following parameters. This is the recommended way
                                                 of configuration.\n\n
                                                 Bit 2 (EIP_OBJECT_QOS_FLAGS_DISABLE_802_1Q):\n
                                                 If set, the stack deactivates attribute 1 of the QoS
                                                 object. So, the 802.1q functionality (VLAN tagging) will
                                                 not be supported. */

  TLR_UINT8     bTag802Enable;              /*!< Enables or disables sending 802.1Q frames on
                                                 CIP messages \n

                                                 \valueRange
                                                  0, 1
                                                 \description
                                                  0:\n
                                                  802.1Q disabled\n\n
                                                  1:\n
                                                  802.1Q enabled \n\n
                                                  Set to 0, if the flag
                                                  EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                  is set in the field ulQoSFlags.*/

  TLR_UINT8     bDSCP_PTP_Event;            /*!< Not used (set to 0) */

  TLR_UINT8     bDSCP_PTP_General;          /*!< Not used (set to 0) */

  TLR_UINT8     bDSCP_Urgent;               /*!< Determines the value of the DSCP field
                                                 for urgent I/O messages. \n

                                                 \valueRange
                                                  0 - 63
                                                 \description
                                                 Set to 0, if the flag
                                                 EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                 is set in the field ulQoSFlags. */

  TLR_UINT8     bDSCP_Scheduled;            /*!< Determines the value of the DSCP field
                                                 for scheduled I/O messages. \n

                                                 \valueRange
                                                  0 - 63
                                                 \description
                                                 Set to 0, if the flag
                                                 EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                 is set in the field ulQoSFlags. */

  TLR_UINT8     bDSCP_High;                 /*!< Determines the value of the DSCP field
                                                 for high priority I/O messages. \n

                                                 \valueRange
                                                  0 - 63
                                                 \description
                                                 Set to 0, if the flag
                                                 EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                 is set in the field ulQoSFlags. */

  TLR_UINT8     bDSCP_Low;                  /*!< Determines the value of the DSCP field
                                                 for low priority I/O messages. \n

                                                 \valueRange
                                                  0 - 63
                                                 \description
                                                 Set to 0, if the flag
                                                 EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                 is set in the field ulQoSFlags. */

  TLR_UINT8     bDSCP_Explicit;             /*!< Determines the value of the DSCP field
                                                 for explicit (acyclic) messages. \n

                                                 \valueRange
                                                  0 - 63
                                                 \description
                                                 Set to 0, if the flag
                                                 EIP_OBJECT_QOS_FLAGS_DEFAULT
                                                 is set in the field ulQoSFlags. */

}__PACKED_POST EIP_OBJECT_CFG_QOS_REQ_T;

/* command for register a new object to the message router */
typedef __PACKED_PRE struct EIP_OBJECT_PACKET_CFG_QOS_REQ_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_CFG_QOS_REQ_T      tData;
}__PACKED_POST EIP_OBJECT_PACKET_CFG_QOS_REQ_T;

#define EIP_OBJECT_CFG_QOS_REQ_SIZE        sizeof(EIP_OBJECT_CFG_QOS_REQ_T)

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_CFG_QOS_CNF_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
}__PACKED_POST EIP_OBJECT_PACKET_CFG_QOS_CNF_T;

#define EIP_OBJECT_CFG_QOS_CNF_SIZE        0

/*! \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_QOS_CHANGE_IND_Content Command related contents
 *  \ingroup EIP_OBJECT_QOS_CHANGE_IND
 * \{ */

typedef __PACKED_PRE struct EIP_OBJECT_QOS_CHANGE_IND_Ttag
{
  TLR_UINT8     bTag802Enable;      /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bTag802Enable
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_PTP_Event;    /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_PTP_Event
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_PTP_General;  /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_PTP_General
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_Urgent;       /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_Urgent
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_Scheduled;    /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_Scheduled
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_High;         /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_High
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_Low;          /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_Low
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
  TLR_UINT8     bDSCP_Explicit;     /*!< See EIP_OBJECT_CFG_QOS_REQ_Ttag::bDSCP_Explicit
                                         from command \ref EIP_OBJECT_CFG_QOS_REQ            */
}__PACKED_POST EIP_OBJECT_QOS_CHANGE_IND_T;

/* command for register a new object to the message router */
typedef __PACKED_PRE struct EIP_OBJECT_PACKET_QOS_CHANGE_IND_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
  EIP_OBJECT_QOS_CHANGE_IND_T   tData;
}__PACKED_POST EIP_OBJECT_PACKET_QOS_CHANGE_IND_T;

#define EIP_OBJECT_QOS_CHANGE_IND_SIZE     sizeof(EIP_OBJECT_QOS_CHANGE_IND_T)

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_QOS_CHANGE_RES_Ttag
{
  TLR_PACKET_HEADER_T           tHead;
}__PACKED_POST EIP_OBJECT_PACKET_QOS_CHANGE_RES_T;

#define EIP_OBJECT_QOS_CHANGE_RES_SIZE     0

/*! \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_TI_SET_SNN_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_TI_SET_SNN_REQ
 *  \{ */

typedef __PACKED_PRE struct EIP_OBJECT_TI_SET_SNN_REQ_Ttag
{
  TLR_UINT8 abSNN[6];   /*!< Safety Network Number */

}__PACKED_POST EIP_OBJECT_TI_SET_SNN_REQ_T;

typedef struct EIP_OBJECT_TI_PACKET_SET_SNN_REQ_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
  EIP_OBJECT_TI_SET_SNN_REQ_T  tData;
}EIP_OBJECT_TI_PACKET_SET_SNN_REQ_T;

#define EIP_OBJECT_TI_SET_SNN_REQ_SIZE     sizeof(EIP_OBJECT_TI_SET_SNN_REQ_T)

typedef struct EIP_OBJECT_TI_PACKET_SET_SNN_CNF_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
}EIP_OBJECT_TI_PACKET_SET_SNN_CNF_T;

#define EIP_OBJECT_TI_SET_SNN_CNF_SIZE     0

/*! \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_SET_PARAMETER_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_SET_PARAMETER_REQ
 *  \{ */

/* Option-Flags for EIP_OBJECT_SET_PARAMETER_REQ */
#define EIP_OBJECT_PRM_FWRD_OPEN_CLOSE_FORWARDING    0x00000001 /*!< Setting this flag the host application will
                                                                     receive the complete forward_open/close message
                                                                     from the stack. \n\n

                                                                     The forward open will be forwarded to the EtherNet/IP
                                                                     Application via the command \ref EIP_OBJECT_CONNECTION_CONFIG_IND.
                                                                     The forward close message will be sent to the EtherNet/IP
                                                                     Application via the command \ref EIP_OBJECT_FORWARD_CLOSE_IND. */

#define EIP_OBJECT_PRM_APPL_TRIG_NO_RPI              0x00000002 /*!< This flag affects the timing of data production
                                                                     in case of "Application Object" or "Change Of State"
                                                                     triggered data.\n
                                                                     Setting this flag turns off the automatic data
                                                                     production using the configured RPI. Data production
                                                                     is then only triggered by the EtherNet/IP Application.     */

#define EIP_OBJECT_PRM_SUPPORT_SNN                    0x00000004 /*!< This flag enables the attribute
                                                                      \"Safety Network Number\" of the TcpIp object.\n
                                                                      (To set the value of the SNN see command
                                                                      \ref EIP_OBJECT_TI_SET_SNN_REQ) */

#define EIP_OBJECT_PRM_ACTIVATE_IDENTITY_RESET_TYPE_1 0x00000008 /*!< This flag enables the additional reset type 1
                                                                      affecting the identity object reset service.
                                                                      Default mode is 0.\n\n
      <b>Default    type 0:</b> This type is supported as default.
                                Emulate as closely as possible cycling power on the item the Identity Object represents. \n\n
      <b>Additional type 1:</b> Return as closely as possible to the factory default configuration, then emulate cycling power as
                                closely as possible. \n\n

      For a proper reset handling please have a look at the command \ref EIP_OBJECT_RESET_IND */

#define EIP_OBJECT_PRM_HARDWARE_CONFIGURABLE          0x00000010 /*!< This flag affects the 2nd attribute of the TCP/IP
                                                                      interface object (class ID 0xF5)
                                                                      If set, the hardware configurable flag within this
                                                                      attribute is set.
                                                                      If not set, the hardware configurable flag within
                                                                      this attribute is not set. */

#define EIP_OBJECT_PRM_SUPPORT_AOT_COS_DATA_PRODUCTION   0x00000020 /*!< This flag enables the "Change of State" (COS) and
                                                                         "Application Object Trigger" feature, which allows the
                                                                         host application to trigger IO data for specific connections
                                                                         via the DPM. */

#define EIP_OBJECT_PRM_SUPPORT_INPUT_ASSEMBLY_STATUS     0x00000040 /*!< This flag enables the status indication functionality.\n\n
                                                                         Every time the status of an assembly instance changes
                                                                         the indication EIP_OBJECT_AS_STATE_IND is generated by the
                                                                         Object Task and is sent to the superior task.\n\n
                                                                         In case of linkable object module, the AP-Task receives this
                                                                         indication and sets up a satus bit list for each input
                                                                         assembly which is the copied into the DPM area. */

#define EIP_OBJECT_PRM_FORWARD_CIP_SERVICE_FOR_UNKNOWN_ASSEMBLY_TO_HOST  0x00000080 /*!< Setting this flag the host application will
                                                                                         receive all CIP service request to assembly instances
                                                                                         that are not registered (indication is done
                                                                                         using command EIP_OBJECT_CL3_SERVICE_IND). */

/* Set Parameter Request */
typedef __PACKED_PRE struct EIP_OBJECT_SET_PARAMETER_REQ_Ttag
{
  TLR_UINT32 ulParameterFlags;    /*!< Parameter flags \n

                                       \description
                                       \ref EIP_OBJECT_PRM_FWRD_OPEN_CLOSE_FORWARDING \n
                                       Enable forwarding of Forward_Open/Forward_Close frames
                                       to the EtherNet/IP Application.

                                       \ref EIP_OBJECT_PRM_APPL_TRIG_NO_RPI \n
                                       Turns off the automatic data production for cyclic I/O messages.

                                       \ref EIP_OBJECT_PRM_SUPPORT_SNN \n
                                       Enable the support of the \" Safety Network Number\" \n

                                       \ref EIP_OBJECT_PRM_ACTIVATE_IDENTITY_RESET_TYPE_1 \n
                                       This flag enables the additional reset type 1
                                       affecting the identity object reset service.

                                       \ref EIP_OBJECT_PRM_HARDWARE_CONFIGURABLE \n
                                       This flag affects the 2nd attribute of the TCP/IP
                                       interface object (class ID 0xF5).\n
                                       If set, the hardware configurable flag within this
                                       attribute is set.
                                        */

}__PACKED_POST EIP_OBJECT_SET_PARAMETER_REQ_T;

typedef struct EIP_OBJECT_PACKET_SET_PARAMETER_REQ_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
  EIP_OBJECT_SET_PARAMETER_REQ_T  tData;
}EIP_OBJECT_PACKET_SET_PARAMETER_REQ_T;

#define EIP_OBJECT_SET_PARAMETER_REQ_SIZE  sizeof(EIP_OBJECT_SET_PARAMETER_REQ_T)

typedef struct EIP_OBJECT_PACKET_SET_PARAMETER_CNF_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
}EIP_OBJECT_PACKET_SET_PARAMETER_CNF_T;

#define EIP_OBJECT_SET_PARAMETER_CNF_SIZE  0

/*! \} */
/*#####################################################################################*/
/*#####################################################################################*/

/* Forward close indication */
typedef __PACKED_PRE struct EIP_OBJECT_FORWARD_CLOSE_IND_Ttag
{
  TLR_UINT8  abData[1]; /* First byte of forward close data */

}__PACKED_POST EIP_OBJECT_FORWARD_CLOSE_IND_T;

typedef struct EIP_OBJECT_PACKET_FORWARD_CLOSE_IND_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
  EIP_OBJECT_FORWARD_CLOSE_IND_T  tData;
}EIP_OBJECT_PACKET_FORWARD_CLOSE_IND_T;

typedef struct EIP_OBJECT_PACKET_FORWARD_CLOSE_RES_Ttag
{
  TLR_PACKET_HEADER_T          tHead;
}EIP_OBJECT_PACKET_FORWARD_CLOSE_RES_T;


/*#####################################################################################*/
/*#####################################################################################*/

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CIP_SERVICE_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_CIP_SERVICE_REQ
 *  \{ */

typedef __PACKED_PRE struct EIP_OBJECT_CIP_SERVICE_REQ_Ttag
{
  TLR_UINT32    ulService;                          /*!< CIP service code          */
  TLR_UINT32    ulClass;                            /*!< CIP class ID              */
  TLR_UINT32    ulInstance;                         /*!< CIP instance number       */
  TLR_UINT32    ulAttribute;                        /*!< CIP attribute number      */
  TLR_UINT8     abData[EIP_OBJECT_MAX_PACKET_LEN];  /*!< CIP Service Data. <br><br>
                                                         Number of bytes provided in this field
                                                         must be add to the packet header length field
                                                         ulLen.
                                                         Do the following to set the proper packet length:<br>
                                                         ptReq->tHead.ulLen = EIP_OBJECT_CIP_SERVICE_REQ_SIZE + number of bytes provided in abData */
}__PACKED_POST EIP_OBJECT_CIP_SERVICE_REQ_T;


typedef struct EIP_OBJECT_PACKET_CIP_SERVICE_REQ_Ttag
{
  TLR_PACKET_HEADER_T            tHead;
  EIP_OBJECT_CIP_SERVICE_REQ_T   tData;
}EIP_OBJECT_PACKET_CIP_SERVICE_REQ_T;

#define EIP_OBJECT_CIP_SERVICE_REQ_SIZE     (sizeof(EIP_OBJECT_CIP_SERVICE_REQ_T) - EIP_OBJECT_MAX_PACKET_LEN)

typedef __PACKED_PRE struct EIP_OBJECT_CIP_SERVICE_CNF_Ttag
{
  TLR_UINT32    ulService;                          /*!< CIP service code          */
  TLR_UINT32    ulClass;                            /*!< CIP class ID              */
  TLR_UINT32    ulInstance;                         /*!< CIP instance number       */
  TLR_UINT32    ulAttribute;                        /*!< CIP attribute number      */

  TLR_UINT32    ulGRC;                              /*!< Generic Error Code        */
  TLR_UINT32    ulERC;                              /*!< Extended Error Code       */

  TLR_UINT8     abData[EIP_OBJECT_MAX_PACKET_LEN];  /*!< CIP service data. <br><br>
                                                         Number of bytes provided in this field
                                                         is encoded in the packet header length field
                                                         ulLen. <br><br>
                                                         Do the following to get the data size:<br>
                                                         ulSize = ptCnf->tHead.ulLen - EIP_OBJECT_CIP_SERVICE_CNF_SIZE */

}__PACKED_POST EIP_OBJECT_CIP_SERVICE_CNF_T;


typedef struct EIP_OBJECT_PACKET_CIP_SERVICE_CNF_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
  EIP_OBJECT_CIP_SERVICE_CNF_T    tData;
}EIP_OBJECT_PACKET_CIP_SERVICE_CNF_T;

#define EIP_OBJECT_CIP_SERVICE_CNF_SIZE     (sizeof(EIP_OBJECT_CIP_SERVICE_CNF_T)) - EIP_OBJECT_MAX_PACKET_LEN

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CIP_OBJECT_CHANGE_IND_Content  Command related contents
 *  \ingroup EIP_OBJECT_CIP_OBJECT_CHANGE_IND
 *  \{ */

/* Signals that the attached data must be stored in non-volatile memory  */
#define EIP_CIP_OBJECT_CHANGE_FLAG_STORE_REMANENT    0x00000001

/* Signal that the object change was done internally. So no service from
   the network has triggered the change indication.
   e.g.: This flag is used when the IP configuration has been applied
         the first time on startup.                                      */
#define EIP_CIP_OBJECT_CHANGE_FLAG_INTERNAL          0x00000002

typedef __PACKED_PRE struct EIP_OBJECT_CIP_OBJECT_CHANGE_IND_Ttag
{
  TLR_UINT32    ulInfoFlags;                        /*!< Information flags     */
  TLR_UINT32    ulService;                          /*!< CIP service code      */
  TLR_UINT32    ulClass;                            /*!< CIP class ID          */
  TLR_UINT32    ulInstance;                         /*!< CIP instance number   */
  TLR_UINT32    ulAttribute;                        /*!< CIP attribute number  */
  TLR_UINT8     abData[EIP_OBJECT_MAX_PACKET_LEN];  /*!< Service Data          */

}__PACKED_POST EIP_OBJECT_CIP_OBJECT_CHANGE_IND_T;


typedef struct EIP_OBJECT_PACKET_CIP_OBJECT_CHANGE_IND_Ttag
{
  TLR_PACKET_HEADER_T                  tHead;
  EIP_OBJECT_CIP_OBJECT_CHANGE_IND_T   tData;
}EIP_OBJECT_PACKET_CIP_OBJECT_CHANGE_IND_T;

#define EIP_OBJECT_CIP_OBJECT_CHANGE_IND_SIZE     (sizeof(EIP_OBJECT_CIP_OBJECT_CHANGE_IND_T) - EIP_OBJECT_MAX_PACKET_LEN)


typedef struct EIP_OBJECT_PACKET_CIP_OBJECT_CHANGE_RES_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
}EIP_OBJECT_PACKET_CIP_OBJECT_CHANGE_RES_T;

#define EIP_OBJECT_CIP_OBJECT_CHANGE_RES_SIZE    0

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ
 *  \{ */

typedef __PACKED_PRE struct EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_Ttag
{
  TLR_UINT32    ulEnable;                           /*!< Specifies activation/deactivation
                                                         0: deactivates attribute
                                                         1: activates attribute    */
  TLR_UINT32    ulClass;                            /*!< CIP class ID              */
  TLR_UINT32    ulInstance;                         /*!< CIP instance number       */
  TLR_UINT32    ulAttribute;                        /*!< CIP attribute number      */

}__PACKED_POST EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_T;


typedef struct EIP_OBJECT_PACKET_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_Ttag
{
  TLR_PACKET_HEADER_T                              tHead;
  EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_T   tData;
}EIP_OBJECT_PACKET_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_T;

#define EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_SIZE     sizeof(EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_REQ_T)


typedef struct EIP_OBJECT_PACKET_CIP_OBJECT_ATTRIBUTE_ACTIVATE_CNF_Ttag
{
  TLR_PACKET_HEADER_T  tHead;
}EIP_OBJECT_PACKET_CIP_OBJECT_ATTRIBUTE_ACTIVATE_CNF_T;

#define EIP_OBJECT_CIP_OBJECT_ATTRIBUTE_ACTIVATE_CNF_SIZE     0

/* \} */
/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_AS_TRIGGER_TYPE_IND_Content  Command related contents
 *  \ingroup EIP_OBJECT_AS_TRIGGER_TYPE_IND
 *  \{ */

#define   EIP_AS_TRIGGER_TYPE_CYCLIC                0x00000001
#define   EIP_AS_TRIGGER_TYPE_CHANGE_OF_STATE       0x00000002
#define   EIP_AS_TRIGGER_TYPE_APPL_OBJ_TRIGGERED    0x00000003

typedef __PACKED_PRE struct EIP_OBJECT_AS_TRIGGER_TYPE_IND_Ttag
{
  TLR_UINT32      ulInstance;       /*!< Assembly instance number             */
  TLR_UINT32      ulDPMOffset;      /*!< DPM offset of the instance data area */
  TLR_UINT32      ulSize;           /*!< Size of the data area                */
  TLR_UINT32      ulFlags;          /*!< Assembly flags                       */

  TLR_UINT32      ulTriggerType;    /*!< Indicates the trigger type that needs
                                         to be used. This information comes
                                         from the connection that is related
                                         the assembly instance.               */

}__PACKED_POST EIP_OBJECT_AS_TRIGGER_TYPE_IND_T;


typedef struct EIP_OBJECT_PACKET_AS_TRIGGER_TYPE_IND_Ttag
{
  TLR_PACKET_HEADER_T                tHead;
  EIP_OBJECT_AS_TRIGGER_TYPE_IND_T   tData;
}EIP_OBJECT_PACKET_AS_TRIGGER_TYPE_IND_T;

#define EIP_OBJECT_AS_TRIGGER_TYPE_IND_SIZE     sizeof(EIP_OBJECT_AS_TRIGGER_TYPE_IND_T)


typedef struct EIP_OBJECT_PACKET_AS_TRIGGER_TYPE_RES_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
}EIP_OBJECT_PACKET_AS_TRIGGER_TYPE_RES_T;

#define EIP_OBJECT_AS_TRIGGER_TYPE_RES_SIZE    0

/* \} */

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_AS_STATE_IND_Content  Command related contents
 *  \ingroup EIP_OBJECT_AS_STATE_IND
 *  \{ */

#define   EIP_AS_STATE_NOT_IN_USED                  0x00000001  /*!< Means: currently not used in an I/O connection */
#define   EIP_AS_STATE_ACTIVE                       0x00000002  /*!< Means: currently used in an I/O connection     */


typedef __PACKED_PRE struct EIP_OBJECT_AS_STATE_IND_Ttag
{
  TLR_UINT32      ulInstance;       /*!< Assembly instance number             */
  TLR_UINT32      ulDPMOffset;      /*!< DPM offset of the instance data area */
  TLR_UINT32      ulSize;           /*!< Size of the data area                */
  TLR_UINT32      ulFlags;          /*!< Assembly flags                       */

  TLR_UINT32      ulState;          /*!< Indicates the state of the instance. */

}__PACKED_POST EIP_OBJECT_AS_STATE_IND_T;


typedef struct EIP_OBJECT_PACKET_AS_STATE_IND_Ttag
{
  TLR_PACKET_HEADER_T         tHead;
  EIP_OBJECT_AS_STATE_IND_T   tData;
}EIP_OBJECT_PACKET_AS_STATE_IND_T;

#define EIP_OBJECT_AS_STATE_IND_SIZE     sizeof(EIP_OBJECT_AS_STATE_IND_T)


typedef struct EIP_OBJECT_PACKET_AS_STATE_RES_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
}EIP_OBJECT_PACKET_AS_STATE_RES_T;

#define EIP_OBJECT_AS_STATE_RES_SIZE    0

/* \} */

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CM_ABORT_CONNECTION_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_CM_ABORT_CONNECTION_REQ
 *  \{ */

typedef __PACKED_PRE struct EIP_OBJECT_CM_ABORT_CONNECTION_REQ_Ttag
{
  TLR_UINT16 usCmInstance;       /*!< Connection manager instance */

}__PACKED_POST EIP_OBJECT_CM_ABORT_CONNECTION_REQ_T;


typedef struct EIP_OBJECT_PACKET_CM_ABORT_CONNECTION_REQ_Ttag
{
  TLR_PACKET_HEADER_T                    tHead;
  EIP_OBJECT_CM_ABORT_CONNECTION_REQ_T   tData;
}EIP_OBJECT_PACKET_CM_ABORT_CONNECTION_REQ_T;

#define EIP_OBJECT_CM_ABORT_CONNECTION_REQ_SIZE     sizeof(EIP_OBJECT_CM_ABORT_CONNECTION_REQ_T)


typedef __PACKED_PRE struct EIP_OBJECT_CM_ABORT_CONNECTION_CNF_Ttag
{
  TLR_UINT16 usCmInstance;       /*!< Connection manager instance */

}__PACKED_POST EIP_OBJECT_CM_ABORT_CONNECTION_CNF_T;


typedef struct EIP_OBJECT_PACKET_CM_ABORT_CONNECTION_CNF_Ttag
{
  TLR_PACKET_HEADER_T                   tHead;
  EIP_OBJECT_CM_ABORT_CONNECTION_CNF_T  tData;
}EIP_OBJECT_PACKET_CM_ABORT_CONNECTION_CNF_T;

#define EIP_OBJECT_CM_ABORT_CONNECTION_CNF_SIZE    sizeof(EIP_OBJECT_CM_ABORT_CONNECTION_CNF_T)

/* \} */

/*#####################################################################################*/
/*#####################################################################################*/

/*!
   Packets defined for forwarding of forward close and forward open
   from Stack to host application.
   Packets are defined on the assumption/rule that the host application uses the received
   packet for generating the response.
   That's why the indication reserves some space that are used for response parameters
*/

typedef __PACKED_PRE struct EIP_CM_APP_FWOPEN_IND_Ttag
{
  TLR_UINT8   bPriority;                            /* Used to calculate request timeout information              */
  TLR_UINT8   bTimeOutTicks;                        /* Used to calculate request timeout information              */
  TLR_UINT32  ulOTConnID;                           /* Network connection ID originator to target                 */
  TLR_UINT32  ulTOConnID;                           /* Network connection ID target to originator                 */
  TLR_UINT16  usConnSerialNum;                      /* Connection serial number                                   */
  TLR_UINT16  usVendorId;                           /* Originator Vendor ID                                       */
  TLR_UINT32  ulOSerialNum;                         /* Originator serial number                                   */
  TLR_UINT8   bTimeoutMultiple;                     /* Connection timeout multiplier                              */
  TLR_UINT8   abReserved1[3];                       /* reserved                                                   */
  TLR_UINT32  ulOTRpi;                              /* Originator to target requested packet rate in microseconds */
  TLR_UINT16  usOTConnParam;                        /* Originator to target connection parameter                  */
  TLR_UINT32  ulTORpi;                              /* target to originator requested packet rate in microseconds */
  TLR_UINT16  usTOConnParam;                        /* target to originator connection parameter                  */
  TLR_UINT8   bTriggerType;                         /* Transport type/trigger                                     */
  TLR_UINT8   bConnPathSize;                        /* Connection path size                                       */
  TLR_UINT8   abConnPath[EIP_OBJECT_MAX_PACKET_LEN]; /* connection path                                           */
}__PACKED_POST EIP_CM_APP_FWOPEN_IND_T;


/* Deliver Forward Open to host application */
typedef __PACKED_PRE struct EIP_OBJECT_FWD_OPEN_FWD_IND_Ttag
{
  TLR_VOID                *pRouteMsg;              /*!< Link to to remember underlying Encapsulation request (must not be modified by app)   */
  TLR_UINT32               aulReserved[4];         /*!< Place holder to be filled by response parameters, see  EIP_OBJECT_FWD_OPEN_FWD_RES_T */
  EIP_CM_APP_FWOPEN_IND_T  tFwdOpenData;           /*!< Forward Open request data to be delivered to host                                    */
}__PACKED_POST EIP_OBJECT_FWD_OPEN_FWD_IND_T;

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_OPEN_FWD_IND_Ttag
{
  TLR_PACKET_HEADER_T                  tHead;
  EIP_OBJECT_FWD_OPEN_FWD_IND_T        tData;
}__PACKED_POST EIP_OBJECT_PACKET_FWD_OPEN_FWD_IND_T;

#define EIP_OBJECT_FWD_OPEN_FWD_IND_SIZE  sizeof(EIP_OBJECT_FWD_OPEN_FWD_IND_T) - EIP_OBJECT_MAX_PACKET_LEN


/*!
   Deliver Forward Open to host application - Response
   Contains the potentially modified forward open data (since host application may need to modify connection points e.g. for safety)
   Additional parameters are: Status (host application result) and a reference (size/offset) to the application reply data
   that needs to be appended to the Forward open response generated by the stack
*/
typedef __PACKED_PRE struct EIP_OBJECT_FWD_OPEN_FWD_RES_Ttag
{
  TLR_VOID*                pRouteMsg;              /*!< Link to underlying Encapsulation request           */
  TLR_UINT32               ulGRC;                  /*!< General Error Code                                 */
  TLR_UINT32               ulERC;                  /*!< Extended Error Code                                */
  TLR_UINT32               ulAppReplyOffset;       /*!< Offset of Application Reply data                   */
  TLR_UINT32               ulAppReplySize;         /*!< Byte-size of Application Reply data                */
  EIP_CM_APP_FWOPEN_IND_T  tFwdOpenData;           /*!< modified forward open (Note that the application
                                                        reply data is appended, which is not visible here) */
}__PACKED_POST EIP_OBJECT_FWD_OPEN_FWD_RES_T;

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_OPEN_FWD_RES_Ttag
{
  TLR_PACKET_HEADER_T                       tHead;
  EIP_OBJECT_FWD_OPEN_FWD_RES_T             tData;
} __PACKED_POST EIP_OBJECT_PACKET_FWD_OPEN_FWD_RES_T;

#define EIP_OBJECT_FWD_OPEN_FWD_RES_SIZE sizeof(EIP_OBJECT_FWD_OPEN_FWD_RES_T) - EIP_OBJECT_MAX_PACKET_LEN

/*#####################################################################################*/
/*#####################################################################################*/

/*! Status indication of Forward Open, that was previously processed by the
 * host (see EIP_OBJECT_FWD_OPEN_FWD_IND)*/
typedef __PACKED_PRE struct EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND_Ttag
{
  TLR_UINT16  usCmInstance;       /*!< Connection manager instance\n\n

                                       \description
                                       The connection is administrated in
                                       the connection manager instance
                                       usCmInstance.

                                       \range
                                       0: Only 0 if connection could not be established\n
                                       1-64: Valid instances              */

  TLR_UINT16  usConnSerialNum;    /*!< Connection serial number           */
  TLR_UINT16  usVendorId;         /*!< Originator Vendor ID               */
  TLR_UINT32  ulOSerialNum;       /*!< Originator serial number           */
  TLR_UINT32  ulGRC;              /*!< General Error Code                 */
  TLR_UINT32  ulERC;              /*!< Extended Error Code                */
}__PACKED_POST EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND_T;

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_OPEN_FWD_COMPLETION_IND_Ttag
{
  TLR_PACKET_HEADER_T                       tHead;
  EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND_T  tData;
}__PACKED_POST EIP_OBJECT_PACKET_FWD_OPEN_FWD_COMPLETION_IND_T;

#define EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND_SIZE   sizeof(EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_IND_T)


typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_OPEN_FWD_COMPLETION_RES_Ttag
{
  TLR_PACKET_HEADER_T                       tHead;
}__PACKED_POST EIP_OBJECT_PACKET_FWD_OPEN_FWD_COMPLETION_RES_T;

#define EIP_OBJECT_FWD_OPEN_FWD_COMPLETION_RES_SIZE   0

/*#####################################################################################*/
/*#####################################################################################*/

typedef __PACKED_PRE struct EIP_CM_APP_FWCLOSE_IND_Ttag
{
  TLR_UINT8   bPriority;                             /* Used to calculate request timeout information  */
  TLR_UINT8   bTimeOutTicks;                         /* Used to calculate request timeout information  */
  TLR_UINT16  usConnSerialNum;                       /* Connection serial number                       */
  TLR_UINT16  usVendorId;                            /* Originator Vendor ID                           */
  TLR_UINT32  ulOSerialNum;                          /* Originator serial number                       */
  TLR_UINT8   bConnPathSize;                         /* Connection path size in 16bit words            */
  TLR_UINT8   bReserved1;
  TLR_UINT8   bConnPath[EIP_OBJECT_MAX_PACKET_LEN];  /* connection path                                */
}__PACKED_POST EIP_CM_APP_FWCLOSE_IND_T;


/* Deliver Forward Close to host application - indication */
typedef __PACKED_PRE struct EIP_OBJECT_FWD_CLOSE_FWD_IND_Ttag
{
  TLR_VOID                 *pRouteMsg;              /*!< Link to underlying Encapsulation request           */
  TLR_UINT32                aulReserved[2];         /*!< Placeholder to be filled by response parameters,
                                                         see  EIP_OBJECT_FWD_CLOSE_FWD_RES_T                */
  EIP_CM_APP_FWCLOSE_IND_T  tFwdCloseData;          /*!< Forward close request data to be delivered to host */
}__PACKED_POST EIP_OBJECT_FWD_CLOSE_FWD_IND_T;

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_CLOSE_FWD_IND_Ttag
{
  TLR_PACKET_HEADER_T             tHead;
  EIP_OBJECT_FWD_CLOSE_FWD_IND_T  tData;
}__PACKED_POST EIP_OBJECT_PACKET_FWD_CLOSE_FWD_IND_T;

#define EIP_OBJECT_FWD_CLOSE_FWD_IND_SIZE   sizeof(EIP_OBJECT_FWD_CLOSE_FWD_IND_T) - EIP_OBJECT_MAX_PACKET_LEN


/*
  Deliver Forward Close to host application - response
  Contains the modified forward close (since application may need to modify connection points e.g. for safety)
  Additional parameters are: Status (host application result)
*/
typedef __PACKED_PRE struct EIP_OBJECT_FWD_CLOSE_FWD_RES_Ttag
{
  TLR_VOID                 *pRouteMsg;             /*!< Link to underlying Encapsulation request */
  TLR_UINT32                ulGRC;                 /*!< General Error Code                       */
  TLR_UINT32                ulERC;                 /*!< Extended Error Code                      */
  EIP_CM_APP_FWCLOSE_IND_T  tFwdCloseData;         /*!< Modified forward close                   */
}__PACKED_POST EIP_OBJECT_FWD_CLOSE_FWD_RES_T;

typedef __PACKED_PRE struct EIP_OBJECT_PACKET_FWD_CLOSE_FWD_RES_Ttag
{
  TLR_PACKET_HEADER_T              tHead;
  EIP_OBJECT_FWD_CLOSE_FWD_RES_T   tData;
} __PACKED_POST EIP_OBJECT_PACKET_FWD_CLOSE_FWD_RES_T;

#define EIP_OBJECT_FWD_CLOSE_FWD_RES_SIZE sizeof(EIP_OBJECT_FWD_CLOSE_FWD_RES_T) - EIP_OBJECT_MAX_PACKET_LEN

/*#####################################################################################*/
/*#####################################################################################*/

/*! Successful forward open response parameters */
typedef __PACKED_PRE struct EIP_SUCCEEDED_FWD_OPEN_RESPONSE_Ttag
{
  TLR_UINT32  ulOTConnID;                             /*!< Network connection ID originator to target     */
  TLR_UINT32  ulTOConnID;                             /*!< Network connection ID target to originator     */
  TLR_UINT16  usConnSerialNum;                        /*!< Connection serial number                       */
  TLR_UINT16  usVendorId;                             /*!< Originator Vendor ID                           */
  TLR_UINT32  ulOSerialNum;                           /*!< Originator serial number                       */
  TLR_UINT32  ulOTApi;                                /*!< Originator to target, actual packet rate       */
  TLR_UINT32  ulTOApi;                                /*!< Target to originator, actual packet rate       */
  TLR_UINT8   bAppReplySize;                          /*!< Number of 16Bit words in the Application Reply */
  TLR_UINT8   bReserved1;                             /*!< Reserved                                       */

  TLR_UINT8   abApplReply[EIP_OBJECT_MAX_PACKET_LEN]; /*!< Application Reply data                         */

}__PACKED_POST  EIP_SUCCEEDED_FWD_OPEN_RESPONSE_T;

/*! Unsuccessful forward open response */
typedef __PACKED_PRE struct EIP_FAILED_FWD_OPEN_RESPONSE_Ttag
{
  TLR_UINT16  usConnSerialNum;    /*!< Connection serial number */
  TLR_UINT16  usVendorId;         /*!< Originator Vendor ID     */
  TLR_UINT32  ulOSerialNum;       /*!< Originator serial number */
  TLR_UINT8   bRemainingPathSize; /*!< Remaining path size      */
  TLR_UINT8   bReserved1;         /*!< Reserved                 */

}__PACKED_POST EIP_FAILED_FWD_OPEN_RESPONSE_T;

typedef __PACKED_PRE struct EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND_Ttag
{
  TLR_UINT16  usCmInstance;       /*!< Connection manager instance\n\n

                                       \description
                                       The connection is administrated in
                                       the connection manager instance
                                       usCmInstance.

                                       \range
                                       0: Only 0 if connection could not be established\n
                                       1-64: Valid instances              */

  TLR_UINT32  ulConnHandle; /*!< Unique connection handle */
  TLR_UINT32  ulGRC;        /*!< General Error Code       */
  TLR_UINT32  ulERC;        /*!< Extended Error Code      */

  TLR_UINT32  ulAdditionalStatusSize; /*!< Number of additional status bytes in abAdditionalStatus */
  TLR_UINT8   abAdditionalStatus[16]; /*!< Additional status                                       */

  __PACKED_PRE union
  {
    EIP_SUCCEEDED_FWD_OPEN_RESPONSE_T    tSuccess; /*!< Forward open response in case of success (ulGRC == 0)     */
    EIP_FAILED_FWD_OPEN_RESPONSE_T       tFail;    /*!< Forward open response in case of an error (ulGRC != 0) */
  } __PACKED_POST tResponse;

}__PACKED_POST EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND_T;


typedef __PACKED_PRE struct EIP_OBJECT_PACKET_CC_FWD_OPEN_RESPONSE_IND_Ttag
{
  TLR_PACKET_HEADER_T                    tHead;
  EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND_T  tData;
}__PACKED_POST EIP_OBJECT_PACKET_CC_FWD_OPEN_RESPONSE_IND_T;

#define EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND_SIZE   sizeof(EIP_OBJECT_CC_FWD_OPEN_RESPONSE_IND_T) - sizeof(EIP_SUCCEEDED_FWD_OPEN_RESPONSE_T)


typedef __PACKED_PRE struct EIP_OBJECT_PACKET_CC_FWD_OPEN_RESPONSE_RES_Ttag
{
  TLR_PACKET_HEADER_T                    tHead;
}__PACKED_POST EIP_OBJECT_PACKET_CC_FWD_OPEN_RESPONSE_RES_T;

#define EIP_OBJECT_CC_FWD_OPEN_RESPONSE_RES_SIZE   0

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_DELETE_IO_CONFIGURATION_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_DELETE_IO_CONFIGURATION_REQ
 *  \{ */

typedef struct EIP_OBJECT_PACKET_DELETE_IO_CONFIGURATION_REQ_Ttag
{
  TLR_PACKET_HEADER_T                    tHead;
}EIP_OBJECT_PACKET_DELETE_IO_CONFIGURATION_REQ_T;

#define EIP_OBJECT_DELETE_IO_CONFIGURATION_REQ_SIZE     0


typedef struct EIP_OBJECT_PACKET_DELETE_IO_CONFIGURATION_CNF_Ttag
{
  TLR_PACKET_HEADER_T                   tHead;
}EIP_OBJECT_PACKET_DELETE_IO_CONFIGURATION_CNF_T;

#define EIP_OBJECT_DELETE_IO_CONFIGURATION_CNF_SIZE    0

/* \} */

/*#####################################################################################*/
/*#####################################################################################*/
/*! \defgroup EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_Content  Command related contents
 *  \ingroup EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ
 *  \{ */

typedef __PACKED_PRE struct EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_Ttag
{
  TLR_UINT32  ulSync0Interval;     /* Sync0 interval in nanoseconds.
                                      Range:
                                          0 (sync0 disabled), or
                                          5us <= ulSync0Interval <= 500ms */

  TLR_UINT32  ulSync0Offset;       /* Sync0 offset in nanoseconds. The offset is related to the second overrun of the PTP time.
                                      Range:
                                          0 <= ulSync0Offset < ulSync0Interval  */

  TLR_UINT32  ulSync1Interval;     /* Sync1 interval in nanoseconds.
                                      Range:
                                          0 (sync1 disabled), or
                                          5us <= ulSync1Interval <= 500ms */

  TLR_UINT32  ulSync1Offset;       /* Sync1 offset in nanoseconds. The offset is related to the second overrun of the PTP time.
                                      Range:
                                          0 <= ulSync10ffset < ulSync1Interval
                                          (in addition, it should have at least a distance of 150ns to ulSync0Offset) */

  TLR_UINT32  ulPulseLength;       /* Pulse length
                                      This parameter specifies the pulse length of sync pins 0 and 1 in microseconds.
                                      Range>
                                          Minimum: 1us
                                          Default: 4us
                                          Maximum: 500, but 1 us smaller than smallest sync interval (sync0, sync1) */

}__PACKED_POST EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_T;

#define EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_SIZE                       sizeof(EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_T)

typedef __PACKED_PRE struct  EIP_OBJECT_PACKET_CREATE_OBJECT_TIMESYNC_REQ_Ttag   /* example request */
{
  TLR_PACKET_HEADER_T                      tHead;
  EIP_OBJECT_CREATE_OBJECT_TIMESYNC_REQ_T  tData;

}__PACKED_POST EIP_OBJECT_PACKET_CREATE_OBJECT_TIMESYNC_REQ_T;


typedef __PACKED_PRE struct  EIP_OBJECT_PACKET_CREATE_OBJECT_TIMESYNC_CNF_Ttag   /* example confirmation */
{
  TLR_PACKET_HEADER_T            tHead;

}__PACKED_POST EIP_OBJECT_PACKET_CREATE_OBJECT_TIMESYNC_CNF_T;

#define EIP_OBJECT_CREATE_OBJECT_TIMESYNC_CNF_SIZE  0

/* \} */

/*#####################################################################################*/
/*#####################################################################################*/

//typedef __TLR_PACKED_PRE struct RCX_GET_DEVICE_INFO_CNF_DATA_Ttag
//{
//  TLR_UINT32 ulDeviceIdx;
//  TLR_UINT32 ulStructId;
//  /* TLR_UINT8  tStruct; Fieldbus specific structure */
//} __TLR_PACKED_POST RCX_GET_DEVICE_INFO_CNF_DATA_T;
//
//typedef struct RCX_GET_DEVICE_INFO_CNF_Ttag
//{
//  TLR_PACKET_HEADER_T            tHead;
//  RCX_GET_DEVICE_INFO_CNF_DATA_T tData;      /* Use type EIP_OBJECT_GET_DEVICE_INFO_CNF_DATA_T instead! (see description below) */
//} __TLR_PACKED_POST RCX_GET_DEVICE_INFO_CNF_T;


/* The following structure can be used when processing the confirmation packet RCX_GET_DEVICE_INFO_CNF.
 * Since there are protocol specific information in the tData part of the confirmation packet,
 * it is easier to use this structure as type of tData. */

typedef __TLR_PACKED_PRE struct EIP_OBJECT_GET_DEVICE_INFO_CNF_DATA_Ttag
{
  RCX_GET_DEVICE_INFO_CNF_DATA_T tDeviceInfo;
  EIP_DEVICE_INFO_T              tSpecificInfo;

}__TLR_PACKED_POST EIP_OBJECT_GET_DEVICE_INFO_CNF_DATA_T;

/*#####################################################################################*/
/*#####################################################################################*/

#define EIP_OBJECT_CHANGE_HOST_TIMEOUT_PACKET_SIZE   0
typedef struct EIP_OBJECT_CHANGE_HOST_TIMEOUT_Ttag{
  TLR_PACKET_HEADER_T         tHead;
} EIP_OBJECT_CHANGE_HOST_TIMEOUT_T;

/*#####################################################################################*/


/* pragma unpack */
#ifdef PRAGMA_PACK_ENABLE
#pragma PRAGMA_UNPACK_1(EIPOBJECT_PUBLIC)
#endif

#endif /* #ifndef __EIPOBJECT_PUBLIC_H */
